package ru.yandex.infra.stage;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.MoreObjects;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.stage.dto.DynamicResourceSpec;
import ru.yandex.infra.stage.dto.RuntimeDeployControls;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.infra.stage.primitives.DeployPrimitiveController.DEPLOY_LABEL_KEY;
import static ru.yandex.infra.stage.primitives.DeployPrimitiveController.DISABLED_CLUSTERS_LABEL_KEY;

// Environment-wide settings, not related to a specific deploy unit
public class StageContext {
    private final String stageFqid;
    private final String stageId;
    private final int revision;
    private final String accountId;
    private final Acl acl;
    private final Map<String, YTreeNode> labels;
    private final long timestamp;
    private final String projectId;
    private final Map<String, DynamicResourceSpec> dynamicResources;
    private final RuntimeDeployControls runtimeDeployControls;
    private final SortedMap<String, String> envVars;
    private final Set<String> disabledClusters;
    private final GlobalContext globalContext;

    public StageContext(String stageFqid,
                        String stageId,
                        int revision,
                        String accountId,
                        Acl acl,
                        long timestamp,
                        String projectId,
                        Map<String, DynamicResourceSpec> dynamicResources,
                        Map<String, YTreeNode> labels,
                        RuntimeDeployControls runtimeDeployControls,
                        Map<String, String> envVars,
                        GlobalContext globalContext) {
        this.stageFqid = stageFqid;
        this.stageId = stageId;
        this.revision = revision;
        this.accountId = accountId;
        this.acl = acl;
        this.timestamp = timestamp;
        this.projectId = projectId;
        this.dynamicResources = dynamicResources;
        this.labels = labels;
        this.runtimeDeployControls = runtimeDeployControls;
        this.envVars = new TreeMap<>(envVars);
        this.globalContext = globalContext;

        this.disabledClusters = getDisabledClusters(labels, globalContext);
    }

    public String getStageFqid() {
        return stageFqid;
    }

    public String getStageId() {
        return stageId;
    }

    public int getRevision() {
        return revision;
    }

    public String getAccountId() {
        return accountId;
    }

    public Acl getAcl() {
        return acl;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public String getProjectId() {
        return projectId;
    }

    public Map<String, DynamicResourceSpec> getDynamicResources() {
        return dynamicResources;
    }

    public RuntimeDeployControls getRuntimeDeployControls() {
        return runtimeDeployControls;
    }

    public SortedMap<String, String> getEnvVars() { return envVars; }

    public Map<String, YTreeNode> getLabels() {
        return labels;
    }

    public Set<String> getDisabledClusters() {
        return disabledClusters;
    }

    public GlobalContext getGlobalContext() {
        return globalContext;
    }

    public StageContext withAcl(Acl acl) {
        return toBuilder().withAcl(acl).build();
    }

    public StageContext withStageFqid(String stageFqid) {
        return toBuilder().withStageFqid(stageFqid).build();
    }

    public StageContext withDynamicResources(Map<String, DynamicResourceSpec> dynamicResources) {
        return toBuilder().withDynamicResources(dynamicResources).build();
    }

    public StageContext withLabels(Map<String, YTreeNode> labels) {
        return toBuilder().withLabels(labels).build();
    }

    public StageContext withRuntimeDeployControls(RuntimeDeployControls runtimeDeployControls) {
        return toBuilder().withRuntimeDeployControls(runtimeDeployControls).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        StageContext that = (StageContext) o;
        return revision == that.revision &&
                timestamp == that.timestamp &&
                Objects.equals(stageFqid, that.stageFqid) &&
                Objects.equals(stageId, that.stageId) &&
                Objects.equals(accountId, that.accountId) &&
                Objects.equals(acl, that.acl) &&
                Objects.equals(projectId, that.projectId) &&
                Objects.equals(dynamicResources, that.dynamicResources) &&
                Objects.equals(labels, that.labels) &&
                Objects.equals(runtimeDeployControls, that.runtimeDeployControls) &&
                Objects.equals(envVars, that.envVars) &&
                Objects.equals(disabledClusters, that.disabledClusters) &&
                Objects.equals(globalContext, that.globalContext);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                stageFqid,
                stageId,
                revision,
                accountId,
                acl,
                timestamp,
                projectId,
                dynamicResources,
                runtimeDeployControls,
                labels,
                envVars,
                disabledClusters,
                globalContext
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("stageFqid", stageFqid)
                .add("stageId", stageId)
                .add("revision", revision)
                .add("accountId", accountId)
                .add("acl", acl)
                .add("timestamp", timestamp)
                .add("projectId", projectId)
                .add("dynamicResources", dynamicResources)
                .add("labels", labels)
                .add("runtimeDeployControls", runtimeDeployControls)
                .add("envVars", envVars)
                .add("disabledClusters", disabledClusters)
                .add("globalContext", globalContext)
                .toString();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private String stageFqid;
        private final String stageId;
        private final int revision;
        private final String accountId;
        private Acl acl;
        private Map<String, YTreeNode> labels;
        private final long timestamp;
        private final String projectId;
        private Map<String, DynamicResourceSpec> dynamicResources;
        private RuntimeDeployControls runtimeDeployControls;
        private final SortedMap<String, String> envVars;
        private final GlobalContext globalContext;

        Builder(StageContext context) {
            this.stageFqid = context.stageFqid;
            this.stageId = context.stageId;
            this.revision = context.revision;
            this.accountId = context.accountId;
            this.acl = context.acl;
            this.timestamp = context.timestamp;
            this.projectId = context.projectId;
            this.dynamicResources = context.dynamicResources;
            this.labels = context.labels;
            this.runtimeDeployControls = context.runtimeDeployControls;
            this.envVars = context.envVars;
            this.globalContext = context.globalContext;
        }

        public StageContext build() {
            return new StageContext(
                    stageFqid,
                    stageId,
                    revision,
                    accountId,
                    acl,
                    timestamp,
                    projectId,
                    dynamicResources,
                    labels,
                    runtimeDeployControls,
                    envVars,
                    globalContext
            );
        }

        Builder withAcl(Acl acl) {
            this.acl = acl;
            return this;
        }

        Builder withStageFqid(String stageFqid) {
            this.stageFqid = stageFqid;
            return this;
        }

        Builder withDynamicResources(Map<String, DynamicResourceSpec> dynamicResources) {
            this.dynamicResources = dynamicResources;
            return this;
        }

        Builder withLabels(Map<String, YTreeNode> labels) {
            this.labels = labels;
            return this;
        }

        Builder withRuntimeDeployControls(RuntimeDeployControls runtimeDeployControls) {
            this.runtimeDeployControls = runtimeDeployControls;
            return this;
        }
    }

    public static Set<String> getDisabledClusters(Map<String, YTreeNode> stageLabels, GlobalContext globalContext) {
        YTreeNode deployLabelNode = stageLabels.get(DEPLOY_LABEL_KEY);
        if (deployLabelNode != null && deployLabelNode.isMapNode()) {
            YTreeNode disabledClustersNode = deployLabelNode.asMap().get(DISABLED_CLUSTERS_LABEL_KEY);
            if (disabledClustersNode != null && disabledClustersNode.isListNode()) {
                final Stream<String> streamFromLabels = disabledClustersNode.asList()
                        .stream()
                        .map(YTreeNode::stringValue);
                return Stream.concat(globalContext.getDisabledClusters().stream(),
                                disabledClustersNode.asList()
                                        .stream()
                                        .map(YTreeNode::stringValue))
                        .collect(Collectors.toSet());
            }
        }
        return globalContext.getDisabledClusters();
    }
}
