package ru.yandex.infra.stage;

import java.util.HashMap;
import java.util.Map;

import io.opentracing.Span;
import io.opentracing.util.GlobalTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.stage.deployunit.DeployUnitController;
import ru.yandex.infra.stage.deployunit.DeployUnitStats;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitStatus;
import ru.yandex.infra.stage.dto.DynamicResourceSpec;
import ru.yandex.infra.stage.dto.DynamicResourceStatus;
import ru.yandex.infra.stage.dto.RuntimeDeployControls;
import ru.yandex.infra.stage.dto.StageSpec;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.util.MapManager;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.DeployObjectId;
import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.stage.protobuf.ConverterUtils.convertMap;

public class StageControllerImpl implements StageController {
    private static final Logger LOG = LoggerFactory.getLogger(StageControllerImpl.class);

    private final DeployUnitControllerFactory factory;
    private final String id;
    private final AclUpdater aclUpdater;
    private final Runnable statusUpdateHandler;
    private final Map<String, DeployUnitController> unitControllers = new HashMap<>();
    private final Map<String, DeployUnitStatus> unitStatuses = new HashMap<>();
    private final Map<String, DynamicResourceStatus> dynamicResourceStatuses = new HashMap<>();
    private final Converter converter;
    private final GlobalContext globalContext;

    public StageControllerImpl(String id,
                               DeployUnitControllerFactory factory,
                               AclUpdater aclUpdater,
                               Runnable statusUpdateHandler,
                               Converter converter,
                               GlobalContext globalContext) {
        this.factory = factory;
        this.id = id;
        this.statusUpdateHandler = statusUpdateHandler;
        this.aclUpdater = aclUpdater;
        this.converter = converter;
        this.globalContext = globalContext;
    }

    @Override
    public void sync(String stageFqid,
                     StageSpec newSpec,
                     RuntimeDeployControls runtimeDeployControls,
                     Map<String, YTreeNode> labels,
                     long newSpecTimestamp,
                     Acl newAcl,
                     String projectId) {
        Span span = GlobalTracer.get()
                .buildSpan("stage-sync")
                .start();
        TStageSpec.Builder stageBuilder = converter.toProto(newSpec).toBuilder();

        StageSpecPatcher runtimeSpecPatcher =
                new DeployUnitsOverridesPatcher(runtimeDeployControls.getDeployUnitOverrides());
        runtimeSpecPatcher.patch(stageBuilder);

        TStageSpec patchedSpec = stageBuilder.build();

        StageContext context = new StageContext(
                stageFqid,
                id,
                patchedSpec.getRevision(),
                patchedSpec.getAccountId(),
                aclUpdater.update(newAcl),
                newSpecTimestamp,
                projectId,
                convertMap(patchedSpec.getDynamicResourcesMap(), converter::fromProto),
                labels,
                runtimeDeployControls,
                patchedSpec.getEnvMap(),
                globalContext
        );

        MapManager.apply(unitControllers, converter.fromProto(patchedSpec).getDeployUnits(),
                (unitId, unitSpec) -> createController(unitId, unitSpec, context),
                (unitId, controller, unitSpec) ->
                        controller.updateSpec(unitSpec, context),
                (unitId, controller) -> {
                    LOG.info("Removing status for removed deploy unit {} for stage '{}'", unitId, id);
                    unitStatuses.remove(unitId);
                    controller.shutdown();
                }, LOG, String.format("deploy unit %s in stage %s", "%s", id));
        span.finish();
    }

    @Override
    public void restoreFromStatus(String stageFqid,
                                  int stageRevision,
                                  Status status,
                                  RuntimeDeployControls runtimeDeployControls, Map<String, YTreeNode> labels,
                                  String accountId,
                                  Acl acl,
                                  String projectId,
                                  Map<String, String> envVars) {
        Span span = GlobalTracer.get()
                .buildSpan("stage-restore")
                .start();
        LOG.info("Restoring controller for stage {}", id);
        Map<String, DynamicResourceSpec> dynamicResources = convertMap(status.getDynamicResources(),
                DynamicResourceStatus::getCurrentTarget);
        status.getDeployUnits().forEach((unitId, unitStatus) -> {
            StageContext context = new StageContext(
                    stageFqid,
                    id,
                    stageRevision,
                    accountId,
                    aclUpdater.update(acl),
                    unitStatus.getTargetSpecTimestamp(),
                    projectId,
                    dynamicResources,
                    labels,
                    runtimeDeployControls,
                    envVars,
                    globalContext
            );
            DeployUnitController controller = unitControllers.get(unitId);
            if (controller == null) {
                controller = createController(unitId, unitStatus.getCurrentTarget(), context);
            }
            controller.restoreFromStatus(unitStatus, context);
            unitControllers.put(unitId, controller);
        });
        LOG.info("Restored controller for stage {}", id);
        span.finish();
    }

    @Override
    public void shutdown() {
        unitControllers.forEach((unitId, controller) -> controller.shutdown());
    }

    @Override
    public Status getStatus() {
        return new Status(new HashMap<>(unitStatuses),
                new HashMap<>(dynamicResourceStatuses)
        );
    }

    @Override
    public Retainment shouldRetain(DeployObjectId objectId, ClusterAndType clusterAndType) {
        String unitId = objectId.getUnitId();
        if (clusterAndType.getType() == YpObjectType.DYNAMIC_RESOURCE || clusterAndType.getType() == YpObjectType.RESOURCE_CACHE) {
            unitId = StringUtils.selectUnitId(unitId);
        }
        DeployUnitController controller = unitControllers.get(unitId);
        if (controller == null) {
            return new Retainment(false, String.format("Deploy unit '%s' is not present in stage '%s'",
                    unitId, id));
        }
        return controller.shouldRetain(clusterAndType);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        unitControllers.forEach((id, controller) -> controller.addStats(builder));
    }

    private String makeFullUnitId(String unitId) {
        return id + StringUtils.ID_SEPARATOR + unitId;
    }

    private DeployUnitController createController(String unitId, DeployUnitSpec unitSpec, StageContext context) {
        LOG.info("Creating controller for deploy unit {} of stage {}", unitId, context.getStageId());
        DeployUnitController controller = factory.createUnitController(unitSpec, unitId, makeFullUnitId(unitId),
                context, status -> handleUpdateUnitStatus(unitId, status),
                this::handleUpdateDynamicResourceStatus,
                (esId, status) -> handleUpdateEndpointSetStatus(esId, unitId, status));
        controller.start();
        return controller;
    }

    private void handleUpdateUnitStatus(String unitId, DeployUnitStatus unitStatus) {
        unitStatuses.put(unitId, unitStatus);
        LOG.debug("Saving new status for unit {} of stage {}", unitId, id);
        sendUpdate();
    }

    private void handleUpdateEndpointSetStatus(String endpointSetId, String unitId, DeployUnitStatus unitStatus) {
        unitStatuses.put(unitId, unitStatus);
        LOG.debug("Saving new status for endpoint set {} in unit {} of stage {}", endpointSetId, unitId, id);
        sendUpdate();
    }

    private void handleUpdateDynamicResourceStatus(String fullId, DynamicResourceStatus status) {
        String resourceId = StringUtils.selectResourceId(fullId);
        if (status == null) {
            dynamicResourceStatuses.remove(resourceId);
        } else {
            dynamicResourceStatuses.put(resourceId, status);
        }
        LOG.debug("Saving new status for dynamic resource {} of stage {}", resourceId, id);
        sendUpdate();
    }

    private void sendUpdate() {
        statusUpdateHandler.run();
    }
}
