package ru.yandex.infra.stage;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.typesafe.config.Config;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.docker.DockerImagesResolverImpl;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.BoxJugglerConfig;
import ru.yandex.infra.stage.dto.CoredumpConfig;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitSpecDetails;
import ru.yandex.infra.stage.dto.LogbrokerConfig;
import ru.yandex.infra.stage.dto.LogbrokerCustomTopicRequest;
import ru.yandex.infra.stage.dto.LogbrokerDestroyPolicy;
import ru.yandex.infra.stage.dto.LogbrokerTopicRequest;
import ru.yandex.infra.stage.dto.McrsUnitSpec;
import ru.yandex.infra.stage.dto.SecretSelector;
import ru.yandex.infra.stage.dto.SidecarVolumeSettings;
import ru.yandex.infra.stage.dto.StageSpec;
import ru.yandex.infra.stage.dto.TvmConfig;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.ThreadLimitsPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils;
import ru.yandex.infra.stage.podspecs.revision.RevisionsHolder;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TMonitoringInfo;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TComputeResources;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TVolume;
import ru.yandex.yp.client.pods.TWorkload;
import ru.yandex.yt.ytree.TAttributeDictionary;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.DISABLE_DISK_ISOLATION_LABEL_ATTRIBUTE;
import static ru.yandex.infra.stage.podspecs.patcher.PatcherUtils.notLogbroker;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.MAX_LOGBROKER_BOX_VCPU;

public class StageValidatorImpl implements StageValidator {
    public static final ImmutableSet<String> TVM_ALL_WORKLOAD_IDS = ImmutableSet.of(
            TvmPatcherUtils.TVM_WORKLOAD_ID
    );
    private static final Pattern ID_PATTERN = Pattern.compile("[A-Za-z0-9-_]+");

    @VisibleForTesting
    static final String DEFAULT_NETWORK_ID_ERROR =
            "Default network id is not specified, need it for infrastructure network setup";

    @VisibleForTesting
    static final String WRONG_ANONYMOUS_MEMORY_ERROR = "Anonymous memory limit (all memory except file caches) should be less than memory limit";

    private static final String VALUE_NOT_IN_SEGMENT_ERROR_FORMAT =
            "Expected %s in [%d, %d], but found %d";

    private static final String VALUES_NOT_EQUAL_ERROR_FORMAT =
            "Expected %s = %s, but found %d != %d";

    @VisibleForTesting
    static final String LOGBROKER_DESTROY_POLICY_ERROR_PREFIX = "Logbroker destroy policy";

    @VisibleForTesting
    static final String LOGBROKER_DESTROY_POLICY_MAX_TRIES_ERROR_FORMAT =
            "max tries should be positive, but found %d";

    @VisibleForTesting
    static final String LOGBROKER_DESTROY_POLICY_RESTART_PERIOD_ERROR_FORMAT =
            "restart period should be at least %d ms, but found %d";

    @VisibleForTesting
    static final String LOGBROKER_DESTROY_POLICY_TOTAL_RESTART_PERIOD_ERROR_FORMAT =
            "total restart period should be at most %d ms, but found %d";

    @VisibleForTesting
    static final String LOGBROKER_CUSTOM_TOPIC_REQUEST_ERROR_PREFIX = "Logbroker custom topic request";

    @VisibleForTesting
    static final String NO_SECRET_FOR_ALIAS_ERROR_FORMAT =
            "There is no secret with alias '%s'";

    @VisibleForTesting
    static final String LOGBROKER_BOX_RESOURCES_REQUEST_ERROR_PREFIX = "Logbroker pod resources request";

    private static final String VCPU = "vcpu", MEMORY = "memory";
    private static final String GUARANTEE = "guarantee", LIMIT = "limit";

    @VisibleForTesting
    static final String VCPU_GUARANTEE = VCPU + " " + GUARANTEE, VCPU_LIMIT = VCPU + " " + LIMIT;

    @VisibleForTesting
    static final String MEMORY_GUARANTEE = MEMORY + " " + GUARANTEE, MEMORY_LIMIT = MEMORY + " " + LIMIT;

    @VisibleForTesting
    static final String MULTIPLE_BOXES_WITH_SAME_JUGGLER_PORTS_FORMAT =
            "All boxes in Deploy Unit must have different Juggler ports. Ports found in multiple boxes: %s";

    @VisibleForTesting
    static final String LOGBROKER_TOOLS_TYPE = "logbroker";

    @VisibleForTesting
    static final String TVM_TOOLS_TYPE = "tvm";

    @VisibleForTesting
    static final String POD_AGENT_SIDECAR_TYPE = "pod_agent";

    private static final String COMMON_DEPLOY_ENV = "Deploy";
    private static final String TVM_ENV = "TVM";

    private final Set<String> knownClusters;
    private final Set<String> blackboxEnvironments;
    private final Optional<String> logbrokerToolsAllocationId;
    private final Optional<String> tvmToolsAllocationId;
    private final Optional<String> podAgentAllocationId;
    private final RevisionsHolder patcherRevisionsHolder;
    private final Config logbrokerDestroyPolicyConfig;

    public StageValidatorImpl(Set<String> knownClusters, Set<String> blackboxEnvironments,
                              Optional<String> logbrokerToolsAllocationId, Optional<String> tvmToolsAllocationId,
                              Optional<String> podAgentAllocationId, RevisionsHolder patcherRevisionsHolder,
                              Config logbrokerDestroyPolicyConfig) {
        this.knownClusters = knownClusters;
        this.blackboxEnvironments = blackboxEnvironments;
        this.logbrokerToolsAllocationId = logbrokerToolsAllocationId;
        this.tvmToolsAllocationId = tvmToolsAllocationId;
        this.podAgentAllocationId = podAgentAllocationId;
        this.patcherRevisionsHolder = patcherRevisionsHolder;
        this.logbrokerDestroyPolicyConfig = logbrokerDestroyPolicyConfig;
    }

    @Override
    public List<String> validate(StageSpec spec, String stageId, Set<String> disabledClusters) {
        List<String> result = new ArrayList<>();
        validatePresenceOfAtLeastOneDeployUnit(spec, result);
        spec.getDeployUnits().forEach((id, unitSpec) -> {
            validateId(id, "Deploy unit id").ifPresent(result::add);
            List<String> unitErrors = validateDeployUnitSpec(unitSpec, stageId, disabledClusters);
            unitErrors.forEach(error -> result.add(String.format("Deploy unit '%s': %s", id, error)));
        });
        return result;
    }

    public static Optional<String> validateId(String id, String description) {
        if (!ID_PATTERN.matcher(id).matches()) {
            return Optional.of(String.format("%s '%s' must match regexp '%s'", description, id, ID_PATTERN.pattern()));
        } else {
            return Optional.empty();
        }
    }

    private static void validatePresenceOfAtLeastOneDeployUnit(StageSpec spec, List<String> result) {
        if (spec.getDeployUnits().isEmpty()) {
            result.add("Stage spec should contain at least one deploy unit");
        }
    }

    List<String> validateDeployUnitSpec(DeployUnitSpec spec, String stageId) {
        return validateDeployUnitSpec(spec, stageId, Collections.emptySet());
    }

    List<String> validateDeployUnitSpec(DeployUnitSpec spec, String stageId, Set<String> disabledClusters) {
        List<String> result = new ArrayList<>();
        if (spec.getNetworkDefaults().getNetworkId().isEmpty()) {
            result.add(DEFAULT_NETWORK_ID_ERROR);
        }

        DeployUnitSpecDetails specDetails = spec.getDetails();
        Set<String> usedClusters = specDetails.extractClusters();
        Set<String> unknownClusters = Sets.difference(usedClusters, knownClusters);
        if (unknownClusters.stream().anyMatch(cluster -> !disabledClusters.contains(cluster))) {
            result.add(String.format("Unknown clusters: %s; allowed are: %s", StringUtils.joinStrings(unknownClusters),
                    StringUtils.joinStrings(knownClusters)));
        }

        if (specDetails instanceof McrsUnitSpec) {
            var clusterList = ((McrsUnitSpec) specDetails).getSpec().getClustersList();
            if (clusterList.size() != usedClusters.size()) {
                result.add(String.format("Duplicated clusters in settings: %s", clusterList.stream()
                        .map(TMultiClusterReplicaSetSpec.TClusterReplicaSetSpecPreferences::getCluster)
                        .collect(Collectors.joining(", ")))
                );
            }
        }

        DataModel.TPodSpec podSpec = specDetails.getPodSpec();

        result.addAll(validateNumUserDisks(specDetails.getLabels(), podSpec));
        result.addAll(validatePodAgentEntitiesVirtualDiskIdRefs(podSpec));

        result.addAll(validateClashWithInfraAllocation(podSpec, logbrokerToolsAllocationId, SidecarToolType.LOGBROKER));
        result.addAll(validateClashWithInfraAllocation(podSpec, tvmToolsAllocationId, SidecarToolType.TVM));
        result.addAll(validateClashWithInfraAllocation(podSpec, podAgentAllocationId, SidecarToolType.POD_AGENT));

        LogbrokerConfig logbrokerConfig = spec.getLogbrokerConfig();
        validateLogbrokerConfig(logbrokerConfig, podSpec, result);

        spec.getTvmConfig().ifPresentOrElse(
                c -> {
                    result.addAll(validateTvmConfig(c));
                    result.addAll(validatePodSpec(podSpec, c.isEnabled(), c.getSidecarVolumeSettings()));
                },
                () -> result.addAll(validatePodSpec(podSpec, false, Optional.empty()))
        );

        TPodAgentSpec podAgentSpec = podSpec.getPodAgentPayload().getSpec();
        if (LogbrokerPatcherUtils.useLogbrokerTools(podAgentSpec, logbrokerConfig)) {
            result.addAll(validateClashWithLogbrokerIds(podAgentSpec, stageId));
            result.addAll(validatePortoLogsVirtualDiskIdRef(logbrokerToolsAllocationId.isPresent(),
                    logbrokerConfig, podSpec.getDiskVolumeRequestsList()));
            result.addAll(validateSidecarVolumeSettings(logbrokerToolsAllocationId.isPresent(),
                    podSpec.getDiskVolumeRequestsList().size(), logbrokerConfig.getSidecarVolumeSettings(),
                    LOGBROKER_TOOLS_TYPE));
        }

        result.addAll(validateSidecarVolumeSettings(podAgentAllocationId.isPresent(),
                podSpec.getDiskVolumeRequestsList().size(),
                specDetails.extractPodAgentConfig().getSidecarVolumeSettings(), POD_AGENT_SIDECAR_TYPE));

        long maxAvailableThreadsPerUserBoxes = ThreadLimitsPatcherUtils.maxAvailableThreadsToUserBoxesForValidation();
        result.addAll(validateThreadLimits(podAgentSpec, maxAvailableThreadsPerUserBoxes));

        validateJugglerConfigs(spec.getBoxJugglerConfigs(), result);
        result.addAll(validateCoredumpConfigs(podAgentSpec, spec.getCoredumpConfig()));

        validatePatchersRevision(spec, patcherRevisionsHolder).ifPresent(result::add);

        specDetails.validateDeploymentStrategy(result);

        spec.getImagesForBoxes().forEach((boxId, description) ->
                DockerImagesResolverImpl.validateDescription(description).forEach(error ->
                        result.add(String.format("Box '%s'. %s", boxId, error))));

        return result;
    }

    List<String> validatePodSpec(DataModel.TPodSpec podSpec, boolean isTvmEnabled,
                                 Optional<SidecarVolumeSettings> tvmToolSidecarVolumeSettings) {
        List<String> result = new ArrayList<>();

        // For now pod agent can understand exactly one 'used_by_infra' disk volume
        // request.
        // We check only 'used_by_infra' label, don't check 'mount_point' and
        // 'volume_type' labels.
        long usedByInfraCount = podSpec.getDiskVolumeRequestsList().stream()
                .filter(r -> PodSpecUtils.hasLabel(r.getLabels(), PodSpecUtils.USED_BY_INFRA_LABEL))
                .count();
        if (usedByInfraCount != 1) {
            result.add(String.format("Pod spec must have exactly 1 " +
                            "'used_by_infra' disk volume request, found: %d",
                    usedByInfraCount));
        }

        if (podSpec.hasPodAgentPayload()) {
            if (podSpec.getPodAgentPayload().hasMeta()) {
                if (podSpec.getPodAgentPayload().getMeta().getBinaryRevision() != 0L) {
                    result.add("Pod agent binary revision in PodAgentDeploymentMeta can't be filled by user");
                }
            }
        }

        validateResourceRequests(podSpec.getResourceRequests(), podSpec.getPodAgentPayload().getSpec(), result);
        validateMonitoringSettings(podSpec.getHostInfra().getMonitoring(), result);
        result.addAll(validatePodAgentSpec(podSpec.getPodAgentPayload().getSpec(), isTvmEnabled));
        if (isTvmEnabled) {
            result.addAll(validateSidecarVolumeSettings(tvmToolsAllocationId.isPresent(),
                    podSpec.getDiskVolumeRequestsList().size(), tvmToolSidecarVolumeSettings, TVM_TOOLS_TYPE));
        }

        return result;
    }

    //check that coredump out volume mounted to box where workload with coredump
    private static List<String> validateCoredumpConfigs(TPodAgentSpec agentSpec,
                                                        Map<String, CoredumpConfig> coredumpConfigs) {
        List<String> result = new ArrayList<>();

        HashMap<String, HashSet<String>> boxIdToCoredumpVolumeIds = new HashMap<>();

        agentSpec.getWorkloadsList().forEach(workload -> {
            CoredumpConfig config = coredumpConfigs.get(workload.getId());
            if (config == null) {
                return;
            }

            config.getOutputPolicy().ifPresent(outPolicy -> {
                if (outPolicy.isOutputVolumeId()) {
                    String boxId = workload.getBoxRef();
                    boxIdToCoredumpVolumeIds.computeIfAbsent(boxId, v -> new HashSet<>()).add(outPolicy.getValue());
                }
            });
        });

        agentSpec.getBoxesList().stream()
                .filter(box -> boxIdToCoredumpVolumeIds.containsKey(box.getId()))
                .forEach(box -> {
                    List<String> boxMountedVolumes =
                            box.getVolumesList().stream().map(TMountedVolume::getVolumeRef).collect(Collectors.toList());
                    HashSet<String> coredumpVolumes = boxIdToCoredumpVolumeIds.get(box.getId());

                    coredumpVolumes.forEach(coredumpVolume -> {
                        if (!boxMountedVolumes.contains(coredumpVolume)) {
                            result.add(String.format("Coredump out volume %s is not mounted to box %s ",
                                    coredumpVolume, box.getId()));
                        }
                    });
                });
        return result;
    }

    private static List<String> validateSidecarVolumeSettings(boolean useDiskAllocationForSidecar,
                                                              long userDisksCount,
                                                              Optional<SidecarVolumeSettings> sidecarVolumeSettings,
                                                              String toolType) {

        List<String> result = new ArrayList<>();
        if (useDiskAllocationForSidecar) {
            //not permitted:
            //unrecognized sidecar volume storage class
            //multiple user disks and auto sidecar volume storage class
            //multiple user disks and not defined sidecar volume storage class(hdd or ssd)

            sidecarVolumeSettings.ifPresentOrElse(volumeSettings -> {
                if (volumeSettings.getStorageClass() == SidecarVolumeSettings.StorageClass.UNRECOGNIZED) {
                    result.add(String.format("storage class is unrecognized for %s sidecar volume settings", toolType));
                } else if (volumeSettings.getStorageClass() == SidecarVolumeSettings.StorageClass.AUTO) {
                    if (userDisksCount > 1) {
                        result.add(String.format("storage class auto is not permitted for %s sidecar volume settings," +
                                " when num user disks is %s", toolType, userDisksCount));
                    }
                }
            }, () -> {
                if (userDisksCount > 1) {
                    result.add(String.format("storage class should be defined for %s sidecar volume settings, when " +
                            "num user disks is %s", toolType, userDisksCount));
                }
            });
        }
        return result;
    }

    private static List<String> validatePortoLogsVirtualDiskIdRef(boolean useDiskAllocationForSidecar,
                                                                  LogbrokerConfig logbrokerConfig,
                                                                  List<DataModel.TPodSpec.TDiskVolumeRequest> userDisks) {
        List<String> result = new ArrayList<>();
        if (useDiskAllocationForSidecar) {
            //not permitted:
            //logs disk id ref not equal to any user disk id
            //logs disk id ref is empty when multiple user disks in spec

            logbrokerConfig.getLogsVirtualDiskIdRef().ifPresentOrElse(logsVirtualDiskId -> {
                if (userDisks.stream().noneMatch(userDisk -> userDisk.getId().equals(logsVirtualDiskId))) {
                    result.add(String.format("Logbroker logs virtual disk id ref: %s not equal to any user disk id",
                            logsVirtualDiskId));
                }
            }, () -> {
                long numUserDisks = userDisks.size();
                if (numUserDisks > 1) {
                    result.add(String.format("Logbroker logs virtual disk id ref should be defined, when num user " +
                            "disks is %s", numUserDisks));
                }
            });
        }
        return result;
    }

    static List<String> validatePodAgentSpec(TPodAgentSpec agentSpec, boolean isTvmEnabled) {
        List<String> result = new ArrayList<>();
        if (isTvmEnabled) {
            result.addAll(validateTVM(agentSpec));
        }

        result.addAll(validateCommonEnv(agentSpec));

        agentSpec.getWorkloadsList().forEach(workload -> {
            Set<String> nameSet = new HashSet<>();
            workload.getEnvList().forEach(env -> {
                String name = env.getName();
                if (nameSet.contains(name)) {
                    result.add(String.format("Multiple entries for environment variable %s in workload %s", name,
                            workload.getId()));
                }
                nameSet.add(name);
            });
        });
        return result;
    }

    private static List<String> validateTVM(TPodAgentSpec agentSpec) {
        List<String> result = new ArrayList<>();

        boolean clashTvmLayer = hasClashedId(
                agentSpec.getResources().getLayersList(),
                TvmPatcherUtils.TVM_BASE_LAYER_ID,
                TLayer::getId);
        if (clashTvmLayer) {
            result.add(String.format("Layer '%s' is reserved for TVM", TvmPatcherUtils.TVM_BASE_LAYER_ID));
        }

        Set<String> clashedTvmResources = findClashedIds(
                agentSpec.getResources().getStaticResourcesList(),
                ImmutableSet.of(TvmPatcherUtils.TVM_CONF_RESOURCE_ID),
                TResource::getId);
        clashedTvmResources.forEach(l -> result.add(String.format("Static resource '%s' is reserved for TVM", l)));

        if (hasClashedId(agentSpec.getBoxesList(), TvmPatcherUtils.TVM_BOX_ID_UNDERSCORED, TBox::getId)) {
            result.add(String.format("Box '%s' is reserved for TVM", TvmPatcherUtils.TVM_BOX_ID_UNDERSCORED));
        }
        if (hasClashedId(agentSpec.getBoxesList(), TvmPatcherUtils.TVM_BOX_ID, TBox::getId)) {
            result.add(String.format("Box '%s' is reserved for TVM", TvmPatcherUtils.TVM_BOX_ID));
        }
        agentSpec.getBoxesList().forEach(box -> {
            validateClashEnv(result, box, TvmPatcherUtils.TVM_LOCAL_TOKEN_ENV, TVM_ENV);
            validateClashEnv(result, box, TvmPatcherUtils.TVM_TOOL_URL_ENV, TVM_ENV);
        });

        Set<String> clashedTvmVolumes = findClashedIds(
                agentSpec.getVolumesList(),
                ImmutableSet.of(TvmPatcherUtils.TVM_LOG_VOLUME_ID, TvmPatcherUtils.TVM_CACHE_VOLUME_ID),
                TVolume::getId);
        clashedTvmVolumes.forEach(l -> result.add(String.format("Volume '%s' is reserved for TVM", l)));

        agentSpec.getWorkloadsList().forEach(workload -> {
            String id = workload.getId();
            if (TVM_ALL_WORKLOAD_IDS.contains(id)) {
                result.add(String.format("Workload '%s' is reserved for TVM", id));
            }
            // Validation of overriding TVM_LOCAL_TOKEN_ENV is not required
            // here, because it will be validated in pod-agent.
            validateClashEnv(result, workload, TvmPatcherUtils.TVM_LOCAL_TOKEN_ENV, TVM_ENV);
            validateClashEnv(result, workload, TvmPatcherUtils.TVM_TOOL_URL_ENV, TVM_ENV);
        });
        return result;
    }

    static List<String> validateCommonEnv(TPodAgentSpec agentSpec) {
        List<String> result = new ArrayList<>();

        agentSpec.getWorkloadsList().stream()
                .filter(w -> notLogbroker(w.getBoxRef()))
                .forEach(workload -> validateClashEnv(result, workload,
                        CommonEnvPatcherUtils.DEPLOY_WORKLOAD_ID_ENV_NAME,
                        COMMON_DEPLOY_ENV));

        agentSpec.getBoxesList().stream()
                .filter(b -> notLogbroker(b.getId()))
                .forEach(box -> {
                    validateClashEnv(result, box, CommonEnvPatcherUtils.DEPLOY_BOX_ID_ENV_NAME, COMMON_DEPLOY_ENV);
                    validateClashEnv(result, box, CommonEnvPatcherUtils.DEPLOY_UNIT_ID_ENV_NAME, COMMON_DEPLOY_ENV);
                    validateClashEnv(result, box, CommonEnvPatcherUtils.DEPLOY_STAGE_ID_ENV_NAME, COMMON_DEPLOY_ENV);
                    validateClashEnv(result, box, CommonEnvPatcherUtils.DEPLOY_PROJECT_ID_ENV_NAME, COMMON_DEPLOY_ENV);
                });

        return result;
    }

    private static List<String> validateNumUserDisks(TAttributeDictionary labels, DataModel.TPodSpec podSpec) {
        List<String> result = new ArrayList<>();

        int numUserDisks = podSpec.getDiskVolumeRequestsList().size();

        boolean specHasDisableDiskIsolationLabel = labels.getAttributesList().stream()
                .anyMatch(a -> a.getKey().equals(DISABLE_DISK_ISOLATION_LABEL_ATTRIBUTE.getKey()));

        if ((numUserDisks > 1) && specHasDisableDiskIsolationLabel) {
            result.add("Only 1 user disk permitted when disk isolation disabled");
        }

        return result;
    }

    private static List<String> validatePodAgentEntitiesVirtualDiskIdRefs(DataModel.TPodSpec podSpec) {
        List<String> result = new ArrayList<>();

        List<String> userDiskIds = podSpec.getDiskVolumeRequestsList().stream()
                .map(DataModel.TPodSpec.TDiskVolumeRequest::getId)
                .collect(Collectors.toList());

        if (userDiskIds.size() > 1) {
            TPodAgentSpec agentSpec = podSpec.getPodAgentPayload().getSpec();
            agentSpec.getBoxesList().forEach(b -> {
                if (!userDiskIds.contains(b.getVirtualDiskIdRef())) {
                    result.add(String.format("Box(%s) has invalid virtual disk id ref(%s)", b.getId(),
                            b.getVirtualDiskIdRef()));
                }
            });

            agentSpec.getVolumesList().forEach(v -> {
                if (!userDiskIds.contains(v.getVirtualDiskIdRef())) {
                    result.add(String.format("Volume(%s) has invalid virtual disk id ref(%s)", v.getId(),
                            v.getVirtualDiskIdRef()));
                }
            });

            agentSpec.getResources().getLayersList().forEach(l -> {
                if (!userDiskIds.contains(l.getVirtualDiskIdRef())) {
                    result.add(String.format("Layer(%s) has invalid virtual disk id ref(%s)", l.getId(),
                            l.getVirtualDiskIdRef()));
                }
            });

            agentSpec.getResources().getStaticResourcesList().forEach(r -> {
                if (!userDiskIds.contains(r.getVirtualDiskIdRef())) {
                    result.add(String.format("Static resource(%s) has invalid virtual disk id ref(%s)", r.getId(),
                            r.getVirtualDiskIdRef()));
                }
            });
        }

        return result;
    }

    private static void validateClashEnv(List<String> result, TWorkload workload, String key, String envBelongsTo) {
        boolean clash = hasClashedId(workload.getEnvList(), key, TEnvVar::getName);
        if (clash) {
            result.add(String.format("Env var '%s' is reserved for %s, but it is found in workload '%s'",
                    key, envBelongsTo, workload.getId()));
        }
    }

    private static void validateClashEnv(List<String> result, TBox box, String key, String envBelongsTo) {
        boolean clash = hasClashedId(box.getEnvList(), key, TEnvVar::getName);
        if (clash) {
            result.add(String.format("Env var '%s' is reserved for %s, but it is found in box '%s'",
                    key, envBelongsTo, box.getId()));
        }
    }

    static List<String> validateClashWithInfraAllocation(DataModel.TPodSpec agentSpec,
                                                         Optional<String> infraAllocationId,
                                                         SidecarToolType sidecarToolType) {
        List<String> result = new ArrayList<>();

        boolean clashWithInfraAllocation =
                infraAllocationId.map(id -> agentSpec.getDiskVolumeRequestsList().stream().anyMatch(request -> request.getId().equals(id))).orElse(false);
        if (clashWithInfraAllocation) {
            result.add(String.format("Allocation id '%s' is reserved for '%s'", infraAllocationId.get(),
                    sidecarToolType.getValue()));
        }

        return result;
    }

    static List<String> validateClashWithLogbrokerIds(TPodAgentSpec agentSpec, String stageId) {

        List<String> result = new ArrayList<>();

        if (!isLogbrokerTestingStage(stageId)) {
            List<String> logbrokerAllWorkloadIds = ImmutableList.of(LogbrokerPatcherUtils.LOGBROKER_AGENT_WORKLOAD_ID,
                    LogbrokerPatcherUtils.LOGBROKER_MONITOR_WORKLOAD_ID);

            Set<String> clashedLogbrokerVolumes = findClashedIds(
                    agentSpec.getVolumesList(),
                    ImmutableSet.of(LogbrokerPatcherUtils.LOGBROKER_AGENT_LOGS_VOLUME_ID,
                            LogbrokerPatcherUtils.LOGBROKER_AGENT_PORTO_LOGS_VOLUME_ID,
                            LogbrokerPatcherUtils.LOGBROKER_AGENT_STATE_VOLUME_ID),
                    TVolume::getId);
            clashedLogbrokerVolumes.forEach(l -> result.add(String.format("Volume '%s' is reserved for logbroker", l)));

            Set<String> clashedLogbrokerLayers = findClashedIds(
                    agentSpec.getResources().getLayersList(),
                    ImmutableSet.of(LogbrokerPatcherUtils.LOGBROKER_BASE_LAYER_ID),
                    TLayer::getId);
            clashedLogbrokerLayers.forEach(l -> result.add(String.format("Layer '%s' is reserved for logbroker", l)));


            agentSpec.getWorkloadsList().forEach(workload -> {
                String id = workload.getId();
                if (logbrokerAllWorkloadIds.contains(id)) {
                    result.add(String.format("Workload '%s' is reserved for logbroker tools", id));
                }
            });

            agentSpec.getWorkloadsList().stream()
                    .filter(TWorkload::getTransmitLogs)
                    .forEach(w -> {
                        validateClashEnv(result, w, LogbrokerPatcherUtils.DEPLOY_LOGNAME_ENV_NAME, COMMON_DEPLOY_ENV);
                        validateClashEnv(result, w, LogbrokerPatcherUtils.DEPLOY_LOGS_ENDPOINT_ENV_NAME,
                                COMMON_DEPLOY_ENV);
                        validateClashEnv(result, w, LogbrokerPatcherUtils.DEPLOY_LOGS_SECRET_ENV_NAME,
                                COMMON_DEPLOY_ENV);
                    });
        }
        return result;
    }

    static List<String> validateThreadLimits(TPodAgentSpec agentSpec, long maxAvailableThreadsPerUserBoxes) {
        List<String> result = new ArrayList<>();

        long threadsNumUsedByUserBoxes = ThreadLimitsPatcherUtils.threadsNumUsedByUserBoxes(agentSpec);

        if (threadsNumUsedByUserBoxes > maxAvailableThreadsPerUserBoxes) {
            result.add(String.format("%d threads used by user boxes is greater than %d - max available threads for " +
                            "user boxes",
                    threadsNumUsedByUserBoxes,
                    maxAvailableThreadsPerUserBoxes));
        }

        long numOfUserBoxesWithoutThreadLimit = ThreadLimitsPatcherUtils.numOfUserBoxesWithoutThreadLimit(agentSpec);
        if (numOfUserBoxesWithoutThreadLimit != 0) {
            long numOfThreadsPossibleToDistribute = maxAvailableThreadsPerUserBoxes - threadsNumUsedByUserBoxes;
            if (numOfThreadsPossibleToDistribute / numOfUserBoxesWithoutThreadLimit < ThreadLimitsPatcherUtils.MIN_NUM_OF_THREADS_PER_USER_BOX) {
                result.add(String.format("Unable to distribute %d threads for %d user boxes, with min %d threads for " +
                                "box",
                        numOfThreadsPossibleToDistribute,
                        numOfUserBoxesWithoutThreadLimit,
                        ThreadLimitsPatcherUtils.MIN_NUM_OF_THREADS_PER_USER_BOX));
            }
        }

        return result;
    }

    private static <T> Set<String> findClashedIds(List<T> objs, Set<String> ids, Function<T, String> idGetter) {
        return objs.stream()
                .map(idGetter)
                .filter(ids::contains)
                .collect(Collectors.toSet());
    }

    private static <T> boolean hasClashedId(List<T> objs, String id, Function<T, String> idGetter) {
        return objs.stream()
                .anyMatch(o -> idGetter.apply(o).equals(id));
    }

    private List<String> validateTvmConfig(TvmConfig tvmConf) {
        List<String> result = new ArrayList<>();
        if (!tvmConf.isEnabled()) {
            return result;
        }
        if (tvmConf.getMode() == TvmConfig.Mode.UNRECOGNIZED) {
            result.add("Unknown mode in TVM config");
        }
        String blackboxEnv = tvmConf.getBlackboxEnvironment();
        if (!blackboxEnvironments.contains(blackboxEnv)) {
            result.add(String.format("Unknown blackbox environment in TVM config: '%s'; allowed are: %s",
                    blackboxEnv, StringUtils.joinStrings(blackboxEnvironments)));
        }
        Set<String> srcAliases = new HashSet<>();
        tvmConf.getClients().forEach((client) -> {
            String srcAlias = client.getSource().getAliasOrAppId();
            if (srcAliases.contains(srcAlias)) {
                result.add(String.format("Source alias '%s' is used multiple times in TVM config",
                        srcAlias));
            } else {
                srcAliases.add(srcAlias);
            }

            Set<String> dstAliases = new HashSet<>();
            client.getDestinations().forEach((dst) -> {
                String dstAlias = dst.getAliasOrAppId();
                if (dstAliases.contains(dstAlias)) {
                    result.add(String.format("Destination alias '%s' is used multiple times for source '%s' in TVM " +
                                    "config",
                            dstAlias, client.getSource().getAppId()));
                } else {
                    dstAliases.add(dstAlias);
                }
            });
        });
        return result;
    }

    private void validateLogbrokerConfig(LogbrokerConfig logbrokerConfig,
                                         DataModel.TPodSpecOrBuilder podSpec,
                                         Collection<String> errors) {
        if (logbrokerConfig.getSidecarBringupMode() == LogbrokerConfig.SidecarBringupMode.UNRECOGNIZED) {
            errors.add("Unrecognized sidecar bring up mode in logbroker config");
        }

        validateLogbrokerDestroyPolicy(logbrokerConfig.getDestroyPolicy(), errors);
        validateLogbrokerTopicRequest(logbrokerConfig.getTopicRequest(), podSpec, errors);
        validateLogbrokerPodResourcesRequest(logbrokerConfig.getPodAdditionalResourcesRequest(), errors);
    }

    private void validateLogbrokerDestroyPolicy(LogbrokerDestroyPolicy destroyPolicy,
                                                Collection<String> errors) {
        int maxTries = destroyPolicy.getMaxTries();
        if (maxTries <= 0) {
            addErrorWithPrefix(
                    errors,
                    LOGBROKER_DESTROY_POLICY_ERROR_PREFIX,
                    String.format(LOGBROKER_DESTROY_POLICY_MAX_TRIES_ERROR_FORMAT, maxTries)
            );
        }

        long restartPeriodMs = destroyPolicy.getRestartPeriodMs();
        long minimalRestartPeriodMs = logbrokerDestroyPolicyConfig.getDuration(
                LogbrokerPatcherUtils.MINIMAL_UNIFIED_AGENT_DESTROY_POLICY_RESTART_PERIOD_MS_CONFIG_KEY
        ).toMillis();

        if (restartPeriodMs < minimalRestartPeriodMs) {
            addErrorWithPrefix(
                    errors,
                    LOGBROKER_DESTROY_POLICY_ERROR_PREFIX,
                    String.format(
                            LOGBROKER_DESTROY_POLICY_RESTART_PERIOD_ERROR_FORMAT,
                            minimalRestartPeriodMs, restartPeriodMs
                    )
            );
        }

        long totalRestartPeriodMs = restartPeriodMs * (maxTries - 1);
        long maximalRestartPeriodMs = logbrokerDestroyPolicyConfig.getDuration(
                LogbrokerPatcherUtils.MAXIMAL_UNIFIED_AGENT_DESTROY_POLICY_TOTAL_RESTART_PERIOD_MS_CONFIG_KEY
        ).toMillis();

        if (totalRestartPeriodMs > maximalRestartPeriodMs) {
            addErrorWithPrefix(
                    errors,
                    LOGBROKER_DESTROY_POLICY_ERROR_PREFIX,
                    String.format(
                            LOGBROKER_DESTROY_POLICY_TOTAL_RESTART_PERIOD_ERROR_FORMAT,
                            maximalRestartPeriodMs, totalRestartPeriodMs
                    )
            );
        }
    }

    private static void validateLogbrokerTopicRequest(LogbrokerTopicRequest topicRequest,
                                                      DataModel.TPodSpecOrBuilder podSpec,
                                                      Collection<String> errors) {
        if (topicRequest instanceof LogbrokerCustomTopicRequest) {
            var customTopicRequest = (LogbrokerCustomTopicRequest) topicRequest;

            var topicDescription = customTopicRequest.getTopicDescription();

            if (0 == topicDescription.getTvmClientId()) {
                addErrorWithPrefix(
                        errors,
                        LOGBROKER_CUSTOM_TOPIC_REQUEST_ERROR_PREFIX,
                        missingOrEmptyErrorMessage("tvm client id")
                );
            }

            if (topicDescription.getName().isEmpty()) {
                addErrorWithPrefix(
                        errors,
                        LOGBROKER_CUSTOM_TOPIC_REQUEST_ERROR_PREFIX,
                        missingOrEmptyErrorMessage("topic name")
                );
            }

            validateSecretSelector(
                    customTopicRequest.getSecretSelector(),
                    podSpec,
                    errors,
                    LOGBROKER_CUSTOM_TOPIC_REQUEST_ERROR_PREFIX
            );
        }
    }

    private static void validateSecretSelector(SecretSelector secretSelector,
                                               DataModel.TPodSpecOrBuilder podSpec,
                                               Collection<String> errors,
                                               String prefix) {
        if (secretSelector.getKey().isEmpty()) {
            addErrorWithPrefix(
                    errors,
                    prefix,
                    missingOrEmptyErrorMessage("secret id")
            );
        }

        var secretAlias = secretSelector.getAlias();
        if (secretAlias.isEmpty()) {
            addErrorWithPrefix(
                    errors,
                    prefix,
                    missingOrEmptyErrorMessage("secret alias")
            );
        } else if (!podSpec.containsSecrets(secretAlias) && !podSpec.containsSecretRefs(secretAlias)) {
            addErrorWithPrefix(
                    errors,
                    prefix,
                    String.format(NO_SECRET_FOR_ALIAS_ERROR_FORMAT, secretAlias)
            );
        }
    }


    private static void validateLogbrokerPodResourcesRequest(
            Optional<AllComputeResources> podAdditionalResourcesRequestOptional,
            Collection<String> errors) {
        podAdditionalResourcesRequestOptional.ifPresent(podAdditionalResourcesRequest -> {
            long vcpuGuarantee = podAdditionalResourcesRequest.getVcpuGuarantee();
            long vcpuLimit = podAdditionalResourcesRequest.getVcpuLimit();

            EntryStream.of(
                Map.of(
                    VCPU_GUARANTEE, vcpuGuarantee,
                    VCPU_LIMIT, vcpuLimit
                )
            ).forKeyValue((vcpuValueName, vcpuValue) ->
                validateValueInSegment(
                        vcpuValueName,
                        vcpuValue,
                        0, MAX_LOGBROKER_BOX_VCPU,
                        errors,
                        LOGBROKER_BOX_RESOURCES_REQUEST_ERROR_PREFIX
                )
            );

            validateValuesNotEqual(
                    VCPU_GUARANTEE, vcpuGuarantee,
                    VCPU_LIMIT, vcpuLimit,
                    errors,
                    LOGBROKER_BOX_RESOURCES_REQUEST_ERROR_PREFIX
            );
        });
    }

    private static void validateResourceRequests(DataModel.TPodSpec.TResourceRequests resources, TPodAgentSpec spec,
                                                 List<String> result) {
        if (resources.getMemoryLimit() == 0) {
            result.add("Memory limit must not be zero");
        }
        if (resources.hasAnonymousMemoryLimit() && resources.getAnonymousMemoryLimit() > resources.getMemoryLimit()) {
            result.add(WRONG_ANONYMOUS_MEMORY_ERROR);
        }
        if (resources.getVcpuLimit() == 0) {
            result.add("Cpu limit must not be zero");
        }
        if (resources.getMemoryLimit() != resources.getMemoryGuarantee()) {
            result.add(String.format("Memory guarantee %s and memory limit %s must be equal",
                    resources.getMemoryGuarantee(), resources.getMemoryLimit()));
        }

        validateBoxesResource(spec, MEMORY_LIMIT, TComputeResources::getMemoryLimit, resources.getMemoryLimit(),
                result);
        validateBoxesResource(spec, MEMORY_GUARANTEE, TComputeResources::getMemoryGuarantee,
                resources.getMemoryGuarantee(), result);
        validateBoxesResource(spec, VCPU_LIMIT, TComputeResources::getVcpuLimit, resources.getVcpuLimit(), result);
        validateBoxesResource(spec, VCPU_GUARANTEE, TComputeResources::getVcpuGuarantee,
                resources.getVcpuGuarantee(), result);
    }

    private static void validateJugglerConfigs(Map<String, BoxJugglerConfig> configs, List<String> result) {
        Map<String, Integer> boxToPort = EntryStream.of(configs)
                .mapValues(config -> config.getPort().orElse(BoxJugglerConfig.DEFAULT_PORT))
                .toMap();
        Map<Integer, List<String>> portToBoxes = boxToPort.entrySet()
                .stream()
                .collect(Collectors.groupingBy(
                        Map.Entry::getValue,
                        Collectors.mapping(Map.Entry::getKey, Collectors.toList())
                ));
        Map<Integer, List<String>> portToMultipleBoxes = EntryStream.of(portToBoxes)
                .filterValues(list -> list.size() > 1)
                .toMap();
        if (!portToMultipleBoxes.isEmpty()) {
            result.add(String.format(MULTIPLE_BOXES_WITH_SAME_JUGGLER_PORTS_FORMAT, portToMultipleBoxes));
        }
    }

    private static void validateMonitoringSettings(TMonitoringInfo monitoring, List<String> result) {
        validateMonitoringLabels(monitoring.getLabelsMap(), result);
        monitoring.getUnistatsList().forEach(endpoint -> validateMonitoringLabels(endpoint.getLabelsMap(), result));
    }

    private static void validateMonitoringLabels(Map<String, String> labels, List<String> result) {
        Sets.intersection(labels.keySet(), MonitoringPatcherUtils.DEPLOY_LABELS)
                .forEach(key -> result.add(String.format("Monitoring label '%s' is reserved by Deploy", key)));
    }

    private static boolean isLogbrokerTestingStage(String stageId) {
        return stageId.startsWith(LogbrokerPatcherUtils.LOGBROKER_TEST_STAGE_PREFIX);
    }

    private static void validateBoxesResource(TPodAgentSpec spec,
                                              String resourceDescription,
                                              Function<TComputeResources, Long> boxResourceSupplier,
                                              Long requestResource,
                                              List<String> result) {
        long boxesResourceSum =
                spec.getBoxesList().stream().mapToLong(box -> boxResourceSupplier.apply(box.getComputeResources())).sum();

        if (boxesResourceSum > requestResource) {
            result.add(String.format("Boxes %s sum %s should be less than %s %s pod",
                    resourceDescription, boxesResourceSum, resourceDescription, requestResource));
        }
    }

    private static Optional<String> validatePatchersRevision(DeployUnitSpec spec, RevisionsHolder revisionsHolder) {
        int patchersRevisionId = spec.getPatchersRevision();
        if (!revisionsHolder.containsRevision(patchersRevisionId)) {
            return Optional.of(String.format("Unknown patchers revision %d", patchersRevisionId));
        }

        return Optional.empty();
    }

    @VisibleForTesting
    static String missingOrEmptyErrorMessage(String fieldName) {
        return String.format("%s missing or empty", fieldName);
    }

    @VisibleForTesting
    static String errorWithPrefix(String prefix, String message) {
        return String.format("%s: %s", prefix, message);
    }

    private static void addErrorWithPrefix(Collection<String> errors, String prefix, String message) {
        errors.add(errorWithPrefix(prefix, message));
    }

    @VisibleForTesting
    static String valueNotInSegmentErrorMessage(String valueName,
                                                long value,
                                                long minValue,
                                                long maxValue) {
        return String.format(
                VALUE_NOT_IN_SEGMENT_ERROR_FORMAT,
                valueName,
                minValue, maxValue, value
        );
    }

    private static void validateValueInSegment(String valueName,
                                               long value,
                                               long minValue,
                                               long maxValue,
                                               Collection<String> errors,
                                               String prefix) {
        if (value < minValue || maxValue < value) {
            String errorMessage = valueNotInSegmentErrorMessage(valueName, value, minValue, maxValue);
            addErrorWithPrefix(
                    errors,
                    prefix,
                    errorMessage
            );
        }
    }

    @VisibleForTesting
    static String valuesNotEqualErrorMessage(String leftValueName,
                                             long leftValue,
                                             String rightValueName,
                                             long rightValue) {
        return String.format(
                VALUES_NOT_EQUAL_ERROR_FORMAT,
                leftValueName, rightValueName,
                leftValue, rightValue
        );
    }

    private static void validateValuesNotEqual(String leftValueName,
                                               long leftValue,
                                               String rightValueName,
                                               long rightValue,
                                               Collection<String> errors,
                                               String prefix) {
        if (leftValue != rightValue) {
            String errorMessage = valuesNotEqualErrorMessage(leftValueName, leftValue, rightValueName, rightValue);
            addErrorWithPrefix(
                    errors,
                    prefix,
                    errorMessage
            );
        }
    }

    enum SidecarToolType {
        LOGBROKER("logbroker"),
        TVM("tvm"),
        POD_AGENT("pod_agent");

        private final String value;

        public String getValue() {
            return value;
        }

        SidecarToolType(String value) {
            this.value = value;
        }
    }

    @VisibleForTesting
    StageValidatorImpl withKnownClusters(Set<String> knownClusters) {
        return toBuilder().withKnownClusters(knownClusters).build();
    }

    @VisibleForTesting
    StageValidatorImpl withBlackboxEnvironments(Set<String> blackboxEnvironments) {
        return toBuilder().withBlackboxEnvironments(blackboxEnvironments).build();
    }

    @VisibleForTesting
    StageValidatorImpl withPatchersRevisionsHolder(RevisionsHolder patcherRevisionsHolder) {
        return toBuilder().withPatchersRevisionsHolder(patcherRevisionsHolder).build();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {

        private Set<String> knownClusters;
        private Set<String> blackboxEnvironments;
        private Optional<String> logbrokerToolsAllocationId;
        private Optional<String> tvmToolsAllocationId;
        private Optional<String> podAgentAllocationId;
        private RevisionsHolder patcherRevisionsHolder;
        private final Config logbrokerDestroyPolicyConfig;

        Builder(StageValidatorImpl validator) {
            this.knownClusters = validator.knownClusters;
            this.blackboxEnvironments = validator.blackboxEnvironments;
            this.logbrokerToolsAllocationId = validator.logbrokerToolsAllocationId;
            this.tvmToolsAllocationId = validator.tvmToolsAllocationId;
            this.podAgentAllocationId = validator.podAgentAllocationId;
            this.patcherRevisionsHolder = validator.patcherRevisionsHolder;
            this.logbrokerDestroyPolicyConfig = validator.logbrokerDestroyPolicyConfig;
        }

        public StageValidatorImpl build() {
            return new StageValidatorImpl(
                    knownClusters,
                    blackboxEnvironments,
                    logbrokerToolsAllocationId,
                    tvmToolsAllocationId,
                    podAgentAllocationId,
                    patcherRevisionsHolder,
                    logbrokerDestroyPolicyConfig
            );
        }

        Builder withKnownClusters(Set<String> knownClusters) {
            this.knownClusters = knownClusters;
            return this;
        }

        Builder withBlackboxEnvironments(Set<String> blackboxEnvironments) {
            this.blackboxEnvironments = blackboxEnvironments;
            return this;
        }

        @VisibleForTesting
        Builder withLogbrokerToolsAllocationId(String logbrokerToolsAllocationId) {
            this.logbrokerToolsAllocationId = Optional.ofNullable(logbrokerToolsAllocationId);
            return this;
        }

        @VisibleForTesting
        Builder withTvmToolsAllocationId(String tvmToolsAllocationId) {
            this.tvmToolsAllocationId = Optional.ofNullable(tvmToolsAllocationId);
            return this;
        }

        @VisibleForTesting
        Builder withPodAgentAllocationId(String podAgentAllocationId) {
            this.podAgentAllocationId = Optional.ofNullable(podAgentAllocationId);
            return this;
        }

        Builder withPatchersRevisionsHolder(RevisionsHolder patcherRevisionsHolder) {
            this.patcherRevisionsHolder = patcherRevisionsHolder;
            return this;
        }
    }
}
