package ru.yandex.infra.stage;

import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.log.Fields;
import io.opentracing.tag.Tags;
import io.opentracing.util.GlobalTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.bolts.function.Function;
import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.stage.deployunit.DeployUnitStats;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.dto.Condition;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.RuntimeDeployControls;
import ru.yandex.infra.stage.dto.StageSpec;
import ru.yandex.infra.stage.dto.StageStatus;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.DeployObjectId;
import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.yp.client.api.AccessControl;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

import static java.lang.String.format;

// Implements common YP object handling (validation, timestamps, check for modification) without any Stage specifics.
public class ValidationControllerImpl implements ValidationController {
    private static final String VALIDATION_FAILED_REASON = "VALIDATION_FAILED";
    private static final String STAGE_SPEC_PARSING_FAILED_REASON = "STAGE_SPEC_PARSING_FAILED";
    private static final String PROJECT_SPEC_PARSING_FAILED_REASON = "PROJECT_SPEC_PARSING_FAILED";
    private static final String PROJECT_ID_DEFAULT_VALUE = "UNKNOWN";

    private static final Logger LOG = LoggerFactory.getLogger(ValidationControllerImpl.class);
    private static final Condition.Status NOT_INIT_STATUS = Condition.Status.UNKNOWN;

    private final String id;
    private final Converter converter;
    private final StageValidator stageValidator;
    private final Clock clock;
    private final StageStatusSender statusSender;
    private final StageController stageController;
    private final GlobalContext globalContext;

    private boolean isInitialized = false;
    private int revision = 0;
    private long timestamp = 0;
    private String storedProjectId = "";
    private Condition validCondition;
    private ValidityType validity;
    private final AclFilter aclFilter;
    private StageStatus lastSendStageStatus;
    private int lastSeenStageRevision;

    public ValidationControllerImpl(StageControllerFactory factory, String stageId, Converter converter,
                                    StageValidator stageValidator, Clock clock, StageStatusSender stageStatusSender,
                                    AclFilter aclFilter, GlobalContext globalContext) {
        this.id = stageId;
        this.converter = converter;
        this.stageValidator = stageValidator;
        this.clock = clock;
        this.statusSender = stageStatusSender;
        this.globalContext = globalContext;

        validCondition = new Condition(NOT_INIT_STATUS, "", "", clock.instant());
        this.stageController = factory.createStageController(id, this::handleUpdate);
        this.aclFilter = aclFilter;
    }

    @Override
    public void sync(Try<YpObject<StageMeta, TStageSpec, TStageStatus>> specMetaStatusTry,
                     Optional<Try<YpObject<SchemaMeta, TProjectSpec, TProjectStatus>>> projectSpecMetaTryOpt) {

        if (specMetaStatusTry.isFailure()) {
            handleParsingFailure(specMetaStatusTry.getThrowable(), STAGE_SPEC_PARSING_FAILED_REASON);
            return;
        }

        YpObject<StageMeta, TStageSpec, TStageStatus> stageObject = specMetaStatusTry.get();

        if (lastSendStageStatus == null) {
            lastSendStageStatus = converter.fromProto(stageObject.getStatus());
        }

        StageMeta stageMeta = stageObject.getMeta();
        TStageSpec stageSpec = stageObject.getSpec();

        if (lastSeenStageRevision != stageSpec.getRevision()) {
            if (lastSeenStageRevision != 0) {
                RootControllerImpl.metricStageSpecUpdatesCount.incrementAndGet();
            }
            lastSeenStageRevision = stageSpec.getRevision();
        }

        if (projectSpecMetaTryOpt.isEmpty()) {
            handleParsingFailure(
                    new Throwable(format("Project %s were not found for stage %s",
                            stageMeta.getProjectId(), stageMeta.getId())),
                    PROJECT_SPEC_PARSING_FAILED_REASON);
            return;
        }
        if (projectSpecMetaTryOpt.get().isFailure()) {
            handleParsingFailure(projectSpecMetaTryOpt.get().getThrowable(), PROJECT_SPEC_PARSING_FAILED_REASON);
            return;
        }

        YpObject<SchemaMeta, TProjectSpec, TProjectStatus> projectObject = projectSpecMetaTryOpt.get().get();

        Acl stageAcl = stageMeta.getAcl();
        Acl projectAcl = projectObject.getMeta().getAcl();

        stageAcl = removeAclOfOldProject(stageAcl, storedProjectId);
        stageAcl = appendProjectAclToStage(stageAcl, projectAcl);

        Tracer tracer = GlobalTracer.get();
        Span span = tracer.buildSpan("stage-validation")
                .withTag("stage_id", stageMeta.getId())
                .withTag("revision", stageObject.getSpec().getRevision())
                .start();
        long newSpecTimestamp = stageObject.getSpecTimestamp();
        try (Scope scope = tracer.scopeManager().activate(span)) {
            StageStatus receivedStatus = converter.fromProto(stageObject.getStatus());
            RuntimeDeployControls runtimeDeployControls =
                    converter.fromProto(stageObject.getStatus().getRuntimeDeployControlsMap());
            StageController.Status innerStatus = StageController.Status.fromStageStatus(receivedStatus);

            StageSpec spec = stageSpec.getAccountId().isEmpty()
                ? converter.fromProto(stageSpec, projectObject.getSpec().getAccountId())
                : converter.fromProto(stageSpec);

            String projectId = stageMeta.getProjectId();
            projectId = projectId.isEmpty() ? PROJECT_ID_DEFAULT_VALUE : projectId;

            if (!isInitialized) {
                LOG.info("Handling initial timestamp {} of stage {}", newSpecTimestamp, id);
                span.setTag("initialization", true);
                validCondition = receivedStatus.getValidated();
                stageController.restoreFromStatus(
                        stageMeta.getFqid(),
                        spec.getRevision(),
                        innerStatus,
                        runtimeDeployControls,
                        stageObject.getLabels(),
                        spec.getAccountId(),
                        stageAcl,
                        projectId,
                        spec.getEnvVars()
                );
                isInitialized = true;
            }

            Set<String> disabledClusters = StageContext.getDisabledClusters(stageObject.getLabels(), globalContext);
            calcValidatedCondition(spec, receivedStatus, disabledClusters);
            if (validCondition.isTrue()) {
                stageController.sync(
                        stageMeta.getFqid(),
                        spec,
                        runtimeDeployControls,
                        stageObject.getLabels(),
                        newSpecTimestamp,
                        stageAcl,
                        projectId
                );

                timestamp = newSpecTimestamp;
                revision = spec.getRevision();

                storedProjectId = projectId;
            }
        } catch (Exception e) {
            LOG.error("Failed to process stage: {}", stageMeta.getId(), e);
            Tags.ERROR.set(span, true);
            span.log(Map.of(Fields.EVENT, "error", Fields.ERROR_OBJECT, e, Fields.MESSAGE, e.toString()));
            handleParsingFailure(e, STAGE_SPEC_PARSING_FAILED_REASON);
        } finally {
            span.finish();
        }
    }

    @Override
    public void shutdown() {
        statusSender.cancelScheduledStatusUpdate(id);
        stageController.shutdown();
    }

    @Override
    public ValidityType getValidConditionType() {
        return validity;
    }

    @Override
    public Retainment shouldRetain(DeployObjectId objectId, ClusterAndType clusterAndType) {
        if (!validCondition.isTrue()) {
            return new Retainment(true, format("Stage '%s' current spec is invalid", id));
        }
        return stageController.shouldRetain(objectId, clusterAndType);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        if (validCondition.isTrue()) {
            stageController.addStats(builder);
        }
    }

    private Acl appendProjectAclToStage(Acl stageAcl, Acl projectAcl) {
        List<AccessControl.TAccessControlEntry> stageEntries = new ArrayList<>(stageAcl.getEntries());
        Set<AccessControl.TAccessControlEntry> stageEntriesSet = new HashSet<>(stageEntries);
        projectAcl.getEntries().stream()
                .filter(entry -> entry.getSubjectsList().stream().anyMatch(aclFilter::isSubjectMatches))
                .map(entry -> filterSubjectsByCondition(entry, aclFilter::isSubjectMatches))
                .filter(entry -> !stageEntriesSet.contains(entry))
                .forEach(stageEntries::add);

        return new Acl(stageEntries);
    }

    private Acl removeAclOfOldProject(Acl stageAcl, String oldProjectId) {
        return new Acl(stageAcl.getEntries().stream()
                .map(entry -> filterSubjectsByCondition(entry,
                        subject -> !aclFilter.isProjectIdMatches(subject, oldProjectId)))
                .filter(entry -> !entry.getSubjectsList().isEmpty())
                .collect(Collectors.toList()));
    }

    private static AccessControl.TAccessControlEntry filterSubjectsByCondition(AccessControl.TAccessControlEntry entry,
                                                                           Function<String, Boolean> condition) {
        return AccessControl.TAccessControlEntry.newBuilder()
                .addAllSubjects(entry.getSubjectsList().stream()
                        .filter(condition::apply)
                        .collect(Collectors.toList()))
                .setAction(entry.getAction())
                .addAllPermissions(entry.getPermissionsList())
                .addAllAttributes(entry.getAttributesList())
                .build();
    }

    private void calcValidatedCondition(StageSpec spec, StageStatus status, Set<String> disabledClusters) {
        List<String> errors = new ArrayList<>();
        List<String> typeChangedUnits = status.getDeployUnits().entrySet().stream().filter(entry -> {
            DeployUnitSpec unitSpec = spec.getDeployUnits().get(entry.getKey());
            return unitSpec != null && unitSpec.getDetails().getClass() !=
                    entry.getValue().getCurrentTarget().getDetails().getClass();
        }).map(Map.Entry::getKey).collect(Collectors.toList());
        if (!typeChangedUnits.isEmpty()) {
            errors.add(format("Deploy unit(s) '%s' changed primitive type", String.join(", ", typeChangedUnits)));
        }

        errors.addAll(stageValidator.validate(spec, id, disabledClusters));
        StageValidatorImpl.validateId(id, "Stage id").ifPresent(errors::add);

        if (errors.isEmpty()) {
            setNewValidCondition(true, "", "");
            validity = ValidityType.VALID;
        } else {
            String userMessage = String.join("; ", errors);
            setNewValidCondition(false, VALIDATION_FAILED_REASON, userMessage);
            validity = ValidityType.INVALID;
            logFailure(VALIDATION_FAILED_REASON, userMessage);
            handleUpdate();
        }
    }

    private void setNewValidCondition(boolean isTrue, String reason, String message) {
        Instant updateTimestamp = validCondition.isTrue() == isTrue ? validCondition.getTimestamp() : clock.instant();
        Condition.Status status = isTrue ? Condition.Status.TRUE : Condition.Status.FALSE;
        validCondition = new Condition(status, reason, message, updateTimestamp);
    }

    private void handleParsingFailure(Throwable error, String reason) {
        String userMessage = ExceptionUtils.getAllMessages(error);
        String logMessage = ExceptionUtils.prettyPrint(error);
        if (validCondition.getStatus() == NOT_INIT_STATUS
                || !validCondition.getReason().equals(STAGE_SPEC_PARSING_FAILED_REASON)
                || !validCondition.getReason().equals(PROJECT_SPEC_PARSING_FAILED_REASON)
                || !validCondition.getMessage().equals(userMessage)) {
            setNewValidCondition(false, reason, userMessage);
            validity = ValidityType.PARSING_FAILED;
            logFailure(reason, logMessage);
            handleUpdate();
        }
    }

    private void logFailure(String reason, String logMessage) {
        LOG.warn("{} for stage {}: {}", reason, id, logMessage);
    }

    private void handleUpdate() {
        //Don't send status updates after intermediate events like "Resolving Sandbox/Docker resources", "RS updated", "RS status changed..." etc
        //Stage status will be updated once at the end of Engine sync cycle.

        //updateStatus();
    }

    @Override
    public void updateStatus() {
        StageStatus newStageStatus = createStatus();
        //comparing dto.StageStatus instead of comparing proto objects (TStageStatus),
        //to be able to ignore timestamps during comparison
        if(!newStageStatus.equals(lastSendStageStatus)) {
            statusSender.save(id, newStageStatus);
            lastSendStageStatus = newStageStatus;
        }
    }

    private StageStatus createStatus() {
        StageController.Status status = stageController.getStatus();
        return new StageStatus(
                status.getDeployUnits(),
                status.getDynamicResources(),
                revision,
                validCondition,
                timestamp
        );
    }
}

