package ru.yandex.infra.stage.cache;

import java.io.FileInputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.google.common.collect.Maps;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public class LocalFsCacheStorage<TProtoValue extends Message> implements CacheStorage<TProtoValue> {
    private static final Logger LOG = LoggerFactory.getLogger(LocalFsCacheStorage.class);
    private static final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "cache"));

    private final Path basePath;
    private final String filename;
    private final Map<String, TProtoValue> values = new ConcurrentHashMap<>();
    private final Supplier<Message.Builder> builderSupplier;

    public LocalFsCacheStorage(String path, CachedObjectType<?, TProtoValue> cachedObjectType, Duration flushInterval) {
        basePath = Path.of(path);
        filename = Path.of(path, cachedObjectType.getName()).toAbsolutePath().toString();
        this.builderSupplier = cachedObjectType.getBuilderSupplier();

        if (!flushInterval.isZero()) {
            executor.scheduleWithFixedDelay(this::flush, flushInterval.toNanos(), flushInterval.toNanos(), TimeUnit.NANOSECONDS);
        }
    }

    @Override
    public CompletableFuture<?> init() {
        try {
            if (!Files.exists(basePath)) {
                LOG.info("Creating directory: {}", basePath.toAbsolutePath());
                Files.createDirectories(basePath);
            }
            if (Files.exists(Path.of(filename))) {
                try (FileInputStream stream = new FileInputStream(filename)) {
                    YTreeNode node = YTreeTextSerializer.deserialize(stream);
                    var map = Maps.transformValues(node.asMap(), n -> {
                        TProtoValue.Builder builder = builderSupplier.get();
                        YTreeProtoUtils.unmarshal(n, builder);
                        return (TProtoValue)builder.build();
                    });
                    values.putAll(map);
                }
            }

        } catch (Exception exception) {
            LOG.error("Failed to initialize cache from file", exception);
            return CompletableFuture.failedFuture(new RuntimeException("Failed to initialize file storage", exception));
        }

        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<?> flush() {
        try {
            YTreeNode node = YTree.builder()
                    .beginMap()
                    .forEach(values.entrySet(),
                            (builder,entry) -> builder.key(entry.getKey())
                                    .value(YTreeProtoUtils.marshal(entry.getValue())))
                    .endMap()
                    .build();
            String data = YTreeTextSerializer.serialize(node);
            Files.writeString(Path.of(filename), data, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
            LOG.info("Cache was successfully flushed to: {}", filename);
        } catch (Exception exception) {
            LOG.error("Failed to flush cache into file", exception);
            return CompletableFuture.failedFuture(new RuntimeException("Failed to flush", exception));
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Map<String, TProtoValue>> read() {
        return CompletableFuture.completedFuture(Collections.unmodifiableMap(values));
    }

    @Override
    public CompletableFuture<?> write(Map<String, TProtoValue> map) {
        values.putAll(map);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<?> write(String key, TProtoValue value) {
        boolean added = values.put(key, value) != null;
        return CompletableFuture.completedFuture(added);
    }

    @Override
    public CompletableFuture<?> remove(String key) {
        boolean removed = values.remove(key) != null;
        return CompletableFuture.completedFuture(removed);
    }
}
