package ru.yandex.infra.stage.concurrent;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.backoff.BackOffExecution;

import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.stage.util.GeneralUtils;

// Executor that performs all actions serially, introducing a global order.
// Acts as main thread.
public class SerialExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(SerialExecutor.class);

    static final String METRIC_SUBMITTED_FUTURES_TOTAL = "submitted_futures_total";
    static final String METRIC_ACTIVE_FUTURES = "active_futures";
    static final String METRIC_FAILED_FUTURES = "failed_futures";
    static final String METRIC_FAILED_FUTURES_RETRIES = "failed_futures_retries";

    static final String METRIC_SUBMITTED_ACTIONS_TOTAL = "submitted_actions_total";
    static final String METRIC_ACTIVE_ACTIONS = "active_actions";
    static final String METRIC_FAILED_ACTIONS = "failed_actions";

    static final String METRIC_CURRENT_ACTION_DELAY = "current_action_delay_ms";
    static final String METRIC_CALLBACKS_TOTAL_EXECUTION_TIME = "callbacks_total_execution_time_ms";

    //Metrics
    private final AtomicLong metricSubmittedFuturesTotal = new AtomicLong();
    private final AtomicInteger metricActiveFutures = new AtomicInteger();
    private final AtomicLong metricFailedFutures = new AtomicLong();
    private final AtomicLong metricFailedFutureRetries = new AtomicLong();
    private final AtomicLong metricSubmittedActionsTotal = new AtomicLong();
    private final AtomicInteger metricActiveActions = new AtomicInteger();
    private final AtomicLong metricFailedActions = new AtomicLong();
    private Stopwatch stopwatchLastActionStart;
    private final AtomicLong metricTimeSpendInCallbacksNanos = new AtomicLong();

    private final ScheduledExecutorService executor;
    private final ConcurrentLinkedQueue<CompletableFuture<?>> submittedFutures = new ConcurrentLinkedQueue<>();

    public SerialExecutor(String threadName, GaugeRegistry metricsRegistry) {
        executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, threadName));
        if (metricsRegistry != null) {
            metricsRegistry.add(METRIC_SUBMITTED_FUTURES_TOTAL, new GolovanableGauge<>(metricSubmittedFuturesTotal::get, "dmmm"));
            metricsRegistry.add(METRIC_ACTIVE_FUTURES, new GolovanableGauge<>(metricActiveFutures::get, "axxx"));
            metricsRegistry.add(METRIC_FAILED_FUTURES, new GolovanableGauge<>(metricFailedFutures::get, "dmmm"));
            metricsRegistry.add(METRIC_FAILED_FUTURES_RETRIES, new GolovanableGauge<>(metricFailedFutureRetries::get, "dmmm"));
            metricsRegistry.add(METRIC_SUBMITTED_ACTIONS_TOTAL, new GolovanableGauge<>(metricSubmittedActionsTotal::get, "dmmm"));
            metricsRegistry.add(METRIC_ACTIVE_ACTIONS, new GolovanableGauge<>(metricActiveActions::get, "axxx"));
            metricsRegistry.add(METRIC_FAILED_ACTIONS, new GolovanableGauge<>(metricFailedActions::get, "dmmm"));
            metricsRegistry.add(METRIC_CALLBACKS_TOTAL_EXECUTION_TIME, new GolovanableGauge<>(() -> metricTimeSpendInCallbacksNanos.get() / 1000000, "dmmm"));

            metricsRegistry.add(METRIC_CURRENT_ACTION_DELAY, new GolovanableGauge<>(() -> {
                Stopwatch stopwatch = stopwatchLastActionStart;
                return stopwatch != null ? stopwatchLastActionStart.elapsed(TimeUnit.MILLISECONDS) : null;
            }, "axxx"));
        }
    }

    public SerialExecutor(String threadName) {
        this(threadName, null);
    }

    // When future completes, execute callback in main thread and return result as a new future.
    public <T> CompletableFuture<T> submitFuture(CompletableFuture<T> future, Consumer<T> onSuccess, Consumer<Throwable> onError) {
        metricSubmittedFuturesTotal.incrementAndGet();
        metricActiveFutures.incrementAndGet();
        final CompletableFuture<T> futureWithResultCallback = future.whenCompleteAsync((result, error) ->
                runWithMetricsInSingleThreadExecutor(() -> {
                    if (error == null) {
                        supressExceptionForCallbackExecution(onSuccess, result);
                    } else {
                        metricFailedFutures.incrementAndGet();
                        supressExceptionForCallbackExecution(onError, error);
                    }
                    metricActiveFutures.decrementAndGet();
                }), executor);
        submittedFutures.add(futureWithResultCallback);
        return futureWithResultCallback;
    }

    public <T> CompletableFuture<T> executeOrRetry(Supplier<CompletableFuture<T>> supplier,
                                                   Consumer<T> onSuccess,
                                                   Consumer<Throwable> onError,
                                                   BackOffExecution backoff) {

        CompletableFuture<T> resultingFuture = new CompletableFuture<>();
        trySubmitNextExecution(resultingFuture, supplier, onSuccess, onError, backoff);
        return resultingFuture;
    }

    public ScheduledFuture<?> schedule(Runnable command, Duration delay) {
        metricSubmittedActionsTotal.incrementAndGet();
        metricActiveActions.incrementAndGet();
        return executor.schedule(() -> {
            try {
                runWithMetricsInSingleThreadExecutor(command);
            }
            catch (Exception exception) {
                metricFailedActions.incrementAndGet();
                LOG.error("Exception while executing scheduled action", exception);
            }
            finally {
                metricActiveActions.decrementAndGet();
            }
        }, delay.toNanos(), TimeUnit.NANOSECONDS);
    }

    public List<CompletableFuture<?>> getAllSubmittedFutures() {
        List<CompletableFuture<?>> result = new ArrayList<>();
        CompletableFuture<?> next;
        while((next = submittedFutures.poll()) != null) {
            result.add(next);
        }
        return result;
    }

    @Deprecated
    public ScheduledExecutorService getExecutor() {
        return executor;
    }

    @VisibleForTesting
    public void shutdown() {
        executor.shutdownNow();
    }

    private <T> void supressExceptionForCallbackExecution(Consumer<T> callback, T arg) {
        try {
            if (callback != null) {
                callback.accept(arg);
            }
        } catch (Exception exception) {
            metricFailedActions.incrementAndGet();
            LOG.error("Exception while executing callback", exception);
        }
    }

    private <T> void trySubmitNextExecution(CompletableFuture<T> resultingFuture,
                                           Supplier<CompletableFuture<T>> supplier,
                                           Consumer<T> onSuccess,
                                           Consumer<Throwable> onError,
                                           BackOffExecution timeout) {
        final Consumer<Throwable> handleErrorAndScheduleNext = error -> {
            supressExceptionForCallbackExecution(onError, error);
            long nextBackOff = timeout.nextBackOff();
            if (nextBackOff == BackOffExecution.STOP) {
                resultingFuture.complete(null);
                return;
            }
            metricFailedFutureRetries.incrementAndGet();
            schedule(() -> this.trySubmitNextExecution(resultingFuture, supplier, onSuccess, onError, timeout),
                    Duration.ofMillis(GeneralUtils.getNextBackoffWithJitter(nextBackOff)));
        };

        CompletableFuture<T> futureToSubmit;
        try {
            futureToSubmit = supplier.get();
        } catch (Exception exception) {
            metricFailedFutures.incrementAndGet();
            LOG.error("Exception while trying to get future for submission", exception);
            handleErrorAndScheduleNext.accept(exception);
            return;
        }

        submitFuture(futureToSubmit, result -> {
            try {
                onSuccess.accept(result);
            } finally {
                resultingFuture.complete(result);
            }
        }, handleErrorAndScheduleNext);
    }

    //Intended to execute only in "executor"'s single thread
    private void runWithMetricsInSingleThreadExecutor(Runnable command) {
        Stopwatch sw = Stopwatch.createStarted();
        stopwatchLastActionStart = sw;
        try {
            command.run();
        } finally {
            metricTimeSpendInCallbacksNanos.addAndGet(sw.elapsed(TimeUnit.NANOSECONDS));
            stopwatchLastActionStart = null;
        }
    }
}
