package ru.yandex.infra.stage.config;

import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.typesafe.config.Config;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.podspecs.PodSpecUtils;

public final class MemoryConfigParser {

    private MemoryConfigParser() {

    }

    @VisibleForTesting
    private static final Map<String, Long> SUFFIX_TO_MULTIPLIER = Map.of(
            "kb", PodSpecUtils.KILOBYTE,
            "mb", PodSpecUtils.MEGABYTE,
            "gb", PodSpecUtils.GIGABYTE
    );

    public static long parseMemoryFrom(Config config, String memoryConfigPath) {
        return parseMemory(config.getString(memoryConfigPath));
    }

    public static long parseMemory(String memoryString) {
        return tryParseWithSuffix(memoryString)
                .orElseGet(() -> Long.parseLong(memoryString));
    }

    private static Optional<Long> tryParseWithSuffix(String memoryString) {
        return EntryStream.of(SUFFIX_TO_MULTIPLIER)
                .filterKeys(memoryString::endsWith)
                .findAny()
                .map(entry -> parseWithSuffix(memoryString, entry.getKey(), entry.getValue()));
    }

    private static long parseWithSuffix(String memoryWithSuffix, String suffix, long multiplier) {
        String valuePrefix = memoryWithSuffix.substring(0, memoryWithSuffix.length() - suffix.length());
        return Long.parseLong(valuePrefix) * multiplier;
    }
}
