package ru.yandex.infra.stage.deployunit;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;

import com.google.common.base.MoreObjects;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.dto.CoredumpConfig;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitSpecDetails;
import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.dto.LogbrokerConfig;
import ru.yandex.infra.stage.dto.LogbrokerTopicConfig;
import ru.yandex.infra.stage.dto.NetworkDefaults;
import ru.yandex.infra.stage.dto.TvmConfig;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public class DeployUnitContext {

    private final StageContext stageContext;
    private final DeployUnitSpec spec;
    private final String fullDeployUnitId;
    private final String deployUnitId;
    private final Map<DockerImageDescription, DockerImageContents> dockerImagesContents;
    private final Optional<LogbrokerTopicConfig> logbrokerTopicConfig;
    private final Map<String, String> resolvedSbr;

    public DeployUnitContext(StageContext stageContext,
                             DeployUnitSpec spec,
                             String deployUnitId,
                             String fullDeployUnitId,
                             Map<DockerImageDescription, DockerImageContents> dockerImagesContents,
                             Optional<LogbrokerTopicConfig> logbrokerTopicConfig,
                             Map<String, String> resolvedSbr) {
        this.stageContext = stageContext;
        this.spec = spec;
        this.fullDeployUnitId = fullDeployUnitId;
        this.deployUnitId = deployUnitId;
        this.dockerImagesContents = dockerImagesContents;
        this.logbrokerTopicConfig = logbrokerTopicConfig;
        this.resolvedSbr = resolvedSbr;
    }

    public StageContext getStageContext() {
        return stageContext;
    }

    public String getFullDeployUnitId() {
        return fullDeployUnitId;
    }

    public String getDeployUnitId() {
        return deployUnitId;
    }

    public Optional<LogbrokerTopicConfig> getLogbrokerTopicConfig() {
        return logbrokerTopicConfig;
    }

    public Map<DockerImageDescription, DockerImageContents> getDockerImagesContents() {
        return dockerImagesContents;
    }

    public Map<String, String> getResolvedSbr() {
        return resolvedSbr;
    }

    public DeployUnitSpec getSpec() {
        return spec;
    }

    public int getPatchersRevision() {
        return spec.getPatchersRevision();
    }

    public DeployUnitContext withStageContext(StageContext stageContext) {
        return toBuilder().withStageContext(stageContext).build();
    }

    public <T> DeployUnitContext withStageContext(BiFunction<StageContext, T, StageContext> stageContextTransformer,
                                                  T transformParameter) {
        return toBuilder().withStageContext(stageContextTransformer, transformParameter).build();
    }

    public DeployUnitContext withLabels(Map<String, YTreeNode> labels) {
        return withStageContext(stageContext.withLabels(labels));
    }

    public DeployUnitContext withLogbrokerTopicConfig(LogbrokerTopicConfig logbrokerTopicConfig) {
        return toBuilder().withLogbrokerTopicConfig(logbrokerTopicConfig).build();
    }

    public DeployUnitContext withDeployUnitSpec(DeployUnitSpec spec) {
        return toBuilder().withSpec(spec).build();
    }

    public <T> DeployUnitContext withDeployUnitSpec(BiFunction<DeployUnitSpec, T, DeployUnitSpec> specTransformer,
                                                    T transformParameter) {
        return toBuilder().withSpec(specTransformer, transformParameter).build();
    }

    public DeployUnitContext withCoredumpConfig(Map<String, CoredumpConfig> coredumpConfig) {
        return withDeployUnitSpec(spec.withCoredumpConfig(coredumpConfig));
    }

    public DeployUnitContext withLogbrokerConfig(LogbrokerConfig logbrokerConfig) {
        return withDeployUnitSpec(spec.withLogbrokerConfig(logbrokerConfig));
    }

    public DeployUnitContext withTvmConfig(TvmConfig tvmConfig) {
        return withDeployUnitSpec(spec.withTvmConfig(tvmConfig));
    }

    public DeployUnitContext withNetworkDefaults(NetworkDefaults networkDefaults) {
        return withDeployUnitSpec(spec.withNetworkDefaults(networkDefaults));
    }

    public DeployUnitContext withDeploySettings(DeployUnitSpec.DeploySettings deploySettings) {
        return withDeployUnitSpec(spec.withDeploySettings(deploySettings));
    }

    public DeployUnitContext withoutDeploySettings() {
        return withDeployUnitSpec(spec.withoutDeploySettings());
    }

    public DeployUnitContext withSpecDetails(DeployUnitSpecDetails specDetails) {
        return withDeployUnitSpec(spec.withDetails(specDetails));
    }

    public DeployUnitContext withDockerImages(Map<DockerImageDescription, DockerImageContents> newDockerImagesContents,
                                              Map<String, DockerImageDescription> newImagesForBoxes) {
        return toBuilder()
                .withSpec(spec.withImagesForBoxes(newImagesForBoxes))
                .withDockerImagesContents(newDockerImagesContents)
                .build();
    }

    public DeployUnitContext withResolvedSbr(Map<String, String> resolvedSbr) {
        return toBuilder().withResolvedSbr(resolvedSbr).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeployUnitContext context = (DeployUnitContext) o;
        return Objects.equals(stageContext, context.stageContext) &&
                Objects.equals(fullDeployUnitId, context.fullDeployUnitId) &&
                Objects.equals(deployUnitId, context.deployUnitId) &&
                Objects.equals(dockerImagesContents, context.dockerImagesContents) &&
                Objects.equals(logbrokerTopicConfig, context.logbrokerTopicConfig) &&
                Objects.equals(resolvedSbr, context.resolvedSbr);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                stageContext,
                spec,
                fullDeployUnitId,
                deployUnitId,
                dockerImagesContents,
                logbrokerTopicConfig,
                resolvedSbr
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("stageContext", stageContext)
                .add("spec", spec)
                .add("fullDeployUnitId", fullDeployUnitId)
                .add("deployUnitId", deployUnitId)
                .add("dockerImagesContents", dockerImagesContents)
                .add("logbrokerTopicConfig", logbrokerTopicConfig)
                .add("resolvedSbr", resolvedSbr)
                .toString();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private StageContext stageContext;
        private DeployUnitSpec spec;
        private final String fullDeployUnitId;
        private final String deployUnitId;
        private Map<DockerImageDescription, DockerImageContents> dockerImagesContents;
        private Optional<LogbrokerTopicConfig> logbrokerTopicConfig;
        private Map<String, String> resolvedSbr;

        public Builder(DeployUnitContext context) {
            this.stageContext = context.stageContext;
            this.spec = context.spec;
            this.fullDeployUnitId = context.fullDeployUnitId;
            this.deployUnitId = context.deployUnitId;
            this.dockerImagesContents = context.dockerImagesContents;
            this.logbrokerTopicConfig = context.logbrokerTopicConfig;
            this.resolvedSbr = context.resolvedSbr;
        }

        public DeployUnitContext build() {
            return new DeployUnitContext(
                    stageContext,
                    spec,
                    deployUnitId,
                    fullDeployUnitId,
                    dockerImagesContents,
                    logbrokerTopicConfig,
                    resolvedSbr
            );
        }

        <T> Builder withStageContext(BiFunction<StageContext, T, StageContext> stageContextTransformer,
                                     T transformParameter) {
            return withStageContext(
                    stageContextTransformer.apply(stageContext, transformParameter)
            );
        }

        Builder withStageContext(StageContext stageContext) {
            this.stageContext = stageContext;
            return this;
        }

        public <T> Builder withSpec(BiFunction<DeployUnitSpec, T, DeployUnitSpec> specTransformer,
                                    T transformParameter) {
            return withSpec(
                    specTransformer.apply(spec, transformParameter)
            );
        }

        Builder withSpec(DeployUnitSpec spec) {
            this.spec = spec;
            return this;
        }

        Builder withLogbrokerTopicConfig(LogbrokerTopicConfig logbrokerTopicConfig) {
            this.logbrokerTopicConfig = Optional.ofNullable(logbrokerTopicConfig);
            return this;
        }

        Builder withDockerImagesContents(Map<DockerImageDescription, DockerImageContents> dockerImagesContents) {
            this.dockerImagesContents = dockerImagesContents;
            return this;
        }

        public Builder withResolvedSbr(Map<String, String> resolvedSbr) {
            this.resolvedSbr = resolvedSbr;
            return this;
        }
    }
}
