package ru.yandex.infra.stage.deployunit;

import java.time.Clock;
import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.google.protobuf.Message;
import io.opentracing.Span;
import io.opentracing.log.Fields;
import io.opentracing.tag.Tags;
import io.opentracing.util.GlobalTracer;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.OptionalF;
import ru.yandex.infra.stage.DeployPrimitiveControllerFactory;
import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.docker.DockerImagesResolver;
import ru.yandex.infra.stage.docker.DockerResolveResultHandler;
import ru.yandex.infra.stage.docker.DockerResolveStatus;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitStatus;
import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.dto.DynamicResourceRevisionStatus;
import ru.yandex.infra.stage.dto.DynamicResourceSpec;
import ru.yandex.infra.stage.dto.DynamicResourceStatus;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils;
import ru.yandex.infra.stage.primitives.AggregatedRawStatus;
import ru.yandex.infra.stage.primitives.DeployPrimitiveController;
import ru.yandex.infra.stage.util.MapManager;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel.TEndpointSetSpec;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceSpec;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerSpec;
import ru.yandex.yp.client.api.TReplicaSetScaleSpec;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.stage.deployunit.Readiness.mergeUnrelated;

public class DeployUnitControllerImpl implements DeployUnitController, DockerResolveResultHandler,
        SandboxResourcesResolveResultHandler {

    @VisibleForTesting
    final static int DEFAULT_LATEST_DEPLOYED_REVISION = 0;

    private static final Logger LOG = LoggerFactory.getLogger(DeployUnitController.class);

    private final String deployUnitId;
    private final String fullDeployUnitId;
    private final MultiplexingController.Factory<TEndpointSetSpec, ReadinessStatus> endpointSetFactory;
    private final MultiplexingController.Factory<TDynamicResourceSpec, DynamicResourceRevisionStatus> dynamicResourceFactory;
    private final Map<String, MultiplexingController<TEndpointSetSpec, ReadinessStatus>> endpointSetMultiplexors = new HashMap<>();
    private final Map<String, MultiplexingController<TDynamicResourceSpec, DynamicResourceRevisionStatus>> dynamicResourcesMultiplexors = new HashMap<>();
    private final MultiplexingController<THorizontalPodAutoscalerSpec, ReadinessStatus> autoscaleMultiplexor;
    private final Consumer<DeployUnitStatus> statusUpdateHandler;
    private final BiConsumer<String, DynamicResourceStatus> dynamicResourceStatusHandler;
    private final BiConsumer<String, DeployUnitStatus> handleUpdateEndpointSetStatus;
    private final LogbrokerTopicConfigResolver logbrokerTopicConfigResolver;
    private final DockerImagesResolver dockerImagesResolver;
    private final DeployPrimitiveControllerFactory factory;
    private Map<DockerImageDescription, DockerImageContents> dockerResolveResult = new HashMap<>();
    private Set<DockerImageDescription> dockerRequiredImages = new HashSet<>();
    private final SandboxResourcesResolver sandboxResolver;
    private final Map<String, TDynamicResourceSpec.Builder> ownDynamicResources = new HashMap<>();

    private DeployPrimitiveController deployController;
    private StageContext stageContext;
    private DeployUnitSpec currentSpec;
    private volatile boolean isLeafObjectsManaged = false;
    private final SpecPatcher<TEndpointSetSpec.Builder> endpointSetSpecPatcher;
    private final SpecPatcher<TDynamicResourceSpec.Builder> dynamicResourceSpecPatcher;


    private final DeployUnitTimelineManager deployUnitTimelineManager;

    private enum UpdateReason {
        START,
        SPEC_UPDATED,
        RESOURCES_UPDATED
    }

    public DeployUnitControllerImpl(DeployUnitSpec spec,
                                    String deployUnitId,
                                    String fullDeployUnitId,
                                    StageContext stageContext,
                                    DeployPrimitiveControllerFactory factory,
                                    MultiplexingController.Factory<TEndpointSetSpec, ReadinessStatus> endpointSetFactory,
                                    MultiplexingController.Factory<TDynamicResourceSpec, DynamicResourceRevisionStatus> dynamicResourceFactory,
                                    MultiplexingController.Factory<THorizontalPodAutoscalerSpec, ReadinessStatus> autoscalerFactory,
                                    DockerImagesResolver dockerImagesResolver,
                                    Consumer<DeployUnitStatus> statusUpdateHandler,
                                    BiConsumer<String, DynamicResourceStatus> dynamicResourceStatusHandler,
                                    BiConsumer<String, DeployUnitStatus> handleUpdateEndpointSetStatus,
                                    Clock clock,
                                    LogbrokerTopicConfigResolver logbrokerTopicConfigResolver,
                                    SandboxResourcesResolver sandboxResolver,
                                    SpecPatcher<TEndpointSetSpec.Builder> endpointSetSpecPatcher, SpecPatcher<TDynamicResourceSpec.Builder> dynamicResourceSpecPatcher) {
        this.currentSpec = spec;
        this.deployUnitId = deployUnitId;
        this.fullDeployUnitId = fullDeployUnitId;
        this.statusUpdateHandler = statusUpdateHandler;
        this.dynamicResourceStatusHandler = dynamicResourceStatusHandler;
        this.handleUpdateEndpointSetStatus = handleUpdateEndpointSetStatus;
        this.deployUnitTimelineManager = new DeployUnitTimelineManager(clock, stageContext.getStageId(), fullDeployUnitId);
        this.logbrokerTopicConfigResolver = logbrokerTopicConfigResolver;
        this.autoscaleMultiplexor = new MultiplexingController<>(fullDeployUnitId, autoscalerFactory, "horizontal pod" +
                " autoscaler",
                (v) -> this.handleNewEvent());
        this.stageContext = stageContext;
        this.dockerImagesResolver = dockerImagesResolver;
        this.dockerImagesResolver.registerResultHandler(fullDeployUnitId, this);
        this.factory = factory;

        this.deployController = factory.createDeployPrimitiveController(spec.getDetails(), this::handleNewEvent,
                fullDeployUnitId, stageContext.getStageFqid());
        this.sandboxResolver = sandboxResolver;

        this.endpointSetFactory = endpointSetFactory;
        this.dynamicResourceFactory = dynamicResourceFactory;
        this.endpointSetSpecPatcher = endpointSetSpecPatcher;
        this.dynamicResourceSpecPatcher = dynamicResourceSpecPatcher;
    }

    @Override
    public void restoreFromStatus(DeployUnitStatus status, StageContext context) {
        Span span = GlobalTracer.get()
                .buildSpan("du-restore")
                .withTag("du_id", fullDeployUnitId)
                .withTag("du_revision", status.getCurrentTarget().getRevision())
                .start();
        LOG.debug("Restoring deploy unit {} from status", fullDeployUnitId);
        deployUnitTimelineManager.restoreFromStatus(status);
        updateSpec(status.getCurrentTarget(), context);
        span.finish();
    }

    @Override
    public void updateSpec(DeployUnitSpec newSpec, StageContext newContext) {
        if (stageContext.getAcl().equals(newContext.getAcl()) &&
                stageContext.getRuntimeDeployControls().equals(newContext.getRuntimeDeployControls(), deployUnitId) &&
                newSpec.getRevision() == currentSpec.getRevision() &&
                stageContext.getStageFqid().equals(newContext.getStageFqid()) &&
                stageContext.getDisabledClusters().equals(newContext.getDisabledClusters())) {
            if (!Objects.equals(stageContext.getDynamicResources(), newContext.getDynamicResources())) {
                LOG.info("Processing deploy unit {} after updates in DynamicResources", fullDeployUnitId);
                stageContext = newContext;
                updateSelf(UpdateReason.RESOURCES_UPDATED);
            }
            return;
        }

        boolean typeChanged = currentSpec.getDetails().getClass() != newSpec.getDetails().getClass();
        currentSpec = newSpec;
        stageContext = newContext;
        if (typeChanged) {
            deployController.shutdown();
            if (!newSpec.getDetails().supportsAutoScaling()) {
                autoscaleMultiplexor.shutdown();
            }

            deployController = factory.createDeployPrimitiveController(newSpec.getDetails(),
                    this::handleNewEvent, fullDeployUnitId, stageContext.getStageFqid());

            LOG.warn("Deploy unit {} changed type and is restarting with revision {} timestamp {}", fullDeployUnitId,
                    newSpec.getRevision(), newContext.getTimestamp());
        } else {
            LOG.info("Updating deploy unit {} to revision {} to timestamp {}", fullDeployUnitId,
                    newSpec.getRevision(),
                    newContext.getTimestamp());
        }

        updateSelf(UpdateReason.SPEC_UPDATED);
    }

    private void manageEndpointSets(DeployUnitContext context) {
        // Constructing full id when id of endpoint set is empty is different because model was changing
        // from alone endpoint set to multiple endpoint set. So, we need to maintain backward compatibility
        Map<String, TEndpointSetSpec.Builder> ownEndpointSets = new HashMap<>(EntryStream.of(currentSpec.getEndpointSets()
                        .entrySet()
                        .stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, v -> v.getValue().toBuilder())))
                .mapKeys(id -> id.isEmpty() ? fullDeployUnitId : childNodeFullId(id))
                .toMap());
        if (!ownEndpointSets.containsKey(fullDeployUnitId)) {
            ownEndpointSets.put(fullDeployUnitId, TEndpointSetSpec.newBuilder());
        }
        manageObjects(ownEndpointSets, endpointSetMultiplexors, endpointSetFactory,
                spec -> endpointSetSpecPatcher.patch(spec, context, new YTreeBuilder()),
                "endpoint set", (id) -> handleUpdateEndpointSetStatus.accept(id, getStatus()));
    }

    private void manageDynamicResources(DeployUnitContext context) {
        ownDynamicResources.clear();
        ownDynamicResources.putAll(EntryStream.of(stageContext.getDynamicResources())
                .filterValues(spec -> spec.getDeployUnitRef().equals(deployUnitId))
                .mapKeys(this::childNodeFullId)
                .mapValues(dynamicResourceSpec -> dynamicResourceSpec.getDynamicResource().toBuilder())
                .toMap());

        manageObjects(ownDynamicResources, dynamicResourcesMultiplexors, dynamicResourceFactory,
                spec -> dynamicResourceSpecPatcher.patch(spec, context, new YTreeBuilder()), "dynamic resource",
                this::handleDynamicResourceEvent);
    }

    private <Builder extends Message.Builder, Spec extends Message, Status extends ReadyStatus> void manageObjects(
            Map<String, Builder> specs,
            Map<String, MultiplexingController<Spec, Status>> controllers,
            MultiplexingController.Factory<Spec, Status> factory,
            Consumer<Builder> specPatcher,
            String objectDescription,
            Consumer<String> handleEvent) {
        MapManager.apply(controllers, specs,
                (id, spec) -> {
                    MultiplexingController<Spec, Status> controller =
                            new MultiplexingController<>(id, factory, objectDescription + "s",
                                    (v) -> handleEvent.accept(id));
                    specPatcher.accept(spec);
                    controller.syncParallel(constructSpecsPerCluster((Spec) spec.build()), stageContext);
                    return controller;
                },
                (id, controller, spec) -> {
                    specPatcher.accept(spec);
                    controller.syncParallel(constructSpecsPerCluster((Spec) spec.build()), stageContext);
                },
                (id, controller) -> {
                    LOG.info("Removing status for removed {} {} for stage '{}'",
                            objectDescription, id, stageContext.getStageId());
                    controller.shutdown();
                    handleEvent.accept(id);
                }, LOG, String.format("%s %s in stage %s", objectDescription, "%s", stageContext.getStageId()));
    }

    @Override
    public Retainment shouldRetain(ClusterAndType clusterAndType) {
        // Until child objects are not under MultiplexingControllers management they should be retained by cluster.
        if (clusterAndType.getType() == YpObjectType.DYNAMIC_RESOURCE && isLeafObjectsManaged) {
            return new Retainment(false, "dynamic resources are already managed");
        }
        if (clusterAndType.getType() == YpObjectType.RESOURCE_CACHE && isLeafObjectsManaged) {
            return new Retainment(false, "resource caches are already managed");
        }
        return currentSpec.getDetails().shouldRetain(clusterAndType);
    }

    private Readiness calcDockerResolveStatus() {
        return dockerRequiredImages.stream().map(description -> dockerImagesResolver.getResolveStatus(description)
                .getReadiness()).reduce(Readiness.ready(), Readiness::mergeUnrelated);
    }

    @VisibleForTesting
    DeployUnitStatus getStatus() {
        AggregatedRawStatus replicaSetStatus = deployController.getStatus();
        if (prerequisitesAreReady()) {

            Map<String, Readiness> endpointSetsStatus = StreamEx.of(endpointSetMultiplexors.keySet())
                    .toMap(id -> ReadinessStatus.merge(endpointSetMultiplexors.get(id).getClusterStatuses()).getReadiness());
            ReadinessStatus autoscalerStatus = ReadinessStatus.merge(autoscaleMultiplexor.getClusterStatuses());

            Readiness unitReadiness = mergeUnrelated(replicaSetStatus.getReadiness(), autoscalerStatus.getReadiness());

            Optional<Readiness> endpointSetsReadinessOpt = endpointSetsStatus.values().stream().reduce(Readiness::mergeUnrelated);
            if (endpointSetsReadinessOpt.isPresent()) {
                unitReadiness = mergeUnrelated(unitReadiness, endpointSetsReadinessOpt.get());
            }
            return createStatus(unitReadiness, replicaSetStatus);
        } else {
            Readiness sandboxResolverStatus = sandboxResolver.getResolveStatus(fullDeployUnitId);
            Readiness dockerResolveStatus = mergeUnrelated(calcDockerResolveStatus(), sandboxResolverStatus);
            Readiness logbrokerStuffStatus = mergeUnrelated(
                    dockerResolveStatus,
                    sandboxResolverStatus);
            Readiness asyncReadiness = isLogsTransmittingEnabled() ? logbrokerStuffStatus : dockerResolveStatus;
            return createStatus(asyncReadiness, replicaSetStatus);
        }
    }

    private DeployUnitDescription getDeployUnitInfo() {
        return new DeployUnitDescription(fullDeployUnitId, currentSpec);
    }

    @Override
    public void start() {
        LOG.info("Starting controller for deploy unit {}", fullDeployUnitId);
        updateSelf(UpdateReason.START);
    }

    @Override
    public void shutdown() {
        LOG.info("Stopping controller for deploy unit {}", fullDeployUnitId);
        isLeafObjectsManaged = false;
        dockerImagesResolver.unregisterResultHandler(fullDeployUnitId);
        sandboxResolver.unregisterResultHandler(fullDeployUnitId);
        endpointSetMultiplexors.values().forEach(MultiplexingController::shutdown);
        autoscaleMultiplexor.shutdown();
        dynamicResourcesMultiplexors.values().forEach(MultiplexingController::shutdown);
        deployController.shutdown();
    }

    private DeployUnitStatus createStatus(Readiness readiness, AggregatedRawStatus status) {
        deployUnitTimelineManager.update(readiness, currentSpec.getRevision());
        Instant timestamp = deployUnitTimelineManager.getTimestamp();
        return new DeployUnitStatus(
                readiness.getInProgressCondition(timestamp),
                readiness.getFailureCondition(timestamp),
                currentSpec,
                currentSpec.getDetails().createStatusDetails(fullDeployUnitId, endpointSetMultiplexors.keySet()
                        .stream().sorted().collect(Collectors.toList()), status.getClusterStatuses()),
                status.getProgress(),
                resolveItype(),
                deployUnitTimelineManager.getDeployUnitTimeline()
        );
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        if (prerequisitesAreReady()) {
            builder.addPrerequisitesReadyDeployUnit();
            deployController.addStats(builder);
            endpointSetMultiplexors.values().forEach(multiplexor -> multiplexor.addStats(builder));
            autoscaleMultiplexor.addStats(builder);
        } else {
            builder.addPrerequisitesUnreadyDeployUnit();
        }
    }

    private <Spec> Map<String, Spec> constructSpecsPerCluster(Spec spec) {
        return StreamEx.of(currentSpec.getDetails().extractClusters())
                .filter(cluster -> !stageContext.getDisabledClusters().contains(cluster))
                .toMap(cluster -> spec);
    }

    private Map<String, THorizontalPodAutoscalerSpec> constructHorizontalPodAutoscaleSpecs() {
        return StreamEx.of(currentSpec.getDetails().extractClusters())
                .mapToEntry(cluster -> currentSpec.getDetails().getAutoscale(cluster))
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .mapValues(this::constructHorizontalPodAutoscaleSpec)
                .toMap();
    }

    private boolean isLogsTransmittingEnabled() {
        return LogbrokerPatcherUtils.useLogbrokerTools(currentSpec.getDetails().getPodSpec().getPodAgentPayload().getSpec(), currentSpec.getLogbrokerConfig());
    }

    private void syncSubordinatesIfReady() {
        if (prerequisitesAreReady()) {
            calcDockerResolverResult();
            DeployUnitContext context = makeContext();
            deployController.sync(currentSpec.getDetails(), context);
            manageEndpointSets(context);
            manageDynamicResources(context);
            if (currentSpec.getDetails().supportsAutoScaling()) {
                autoscaleMultiplexor.syncParallel(constructHorizontalPodAutoscaleSpecs(), stageContext);
            }
            isLeafObjectsManaged = true;
        }
    }

    private void updateSelf(UpdateReason reason) {
        Span span = GlobalTracer.get().buildSpan("du-update")
                .withTag("reason", reason.name())
                .start();
        try (var ignored = GlobalTracer.get().scopeManager().activate(span)) {
            updateDocker(reason);
            updateSandbox();
            syncSubordinatesIfReady();
            handleNewEvent();
        } catch (Exception e) {
            LOG.error("Failed to process deployUnit update: {}", fullDeployUnitId, e);
            Tags.ERROR.set(span, true);
            span.log(Map.of(Fields.EVENT, "error", Fields.ERROR_OBJECT, e, Fields.MESSAGE, e.toString()));
        } finally {
            span.finish();
        }
    }

    private void handleNewEvent() {
        statusUpdateHandler.accept(getStatus());
    }

    // DeployUnitController should pass an actual dynamic resource/resource cache spec and aggregated cluster status to StageController
    // which will write statuses to yp. This action is required because TStageDynamicResourceStatus
    // are not a part of TDeployUnitStatus
    private void handleDynamicResourceEvent(String resourceId) {
        if (ownDynamicResources.containsKey(resourceId)) {

            dynamicResourceStatusHandler.accept(resourceId,
                    new DynamicResourceStatus(new DynamicResourceSpec(deployUnitId,
                            ownDynamicResources.get(resourceId).build()),
                            DynamicResourceRevisionStatus.aggregateFromClusters(dynamicResourcesMultiplexors.get(resourceId).getClusterStatuses())));
        } else {
            dynamicResourceStatusHandler.accept(resourceId, null);
        }
    }

    @Override
    public void onDockerResolveSuccess(DockerImageDescription description, DockerImageContents result) {
        syncSubordinatesIfReady();
        handleNewEvent();
    }

    @Override
    public void onDockerResolveFailure(DockerImageDescription description, Throwable ex) {
        LOG.warn("Docker resolver received error for {} for image {}: {}",
                fullDeployUnitId, description, ex.getMessage());
        handleNewEvent();
    }

    @Override
    public void onSandboxResourceResolveSuccess() {
        syncSubordinatesIfReady();
        handleNewEvent();
    }

    @Override
    public void onSandboxResourceResolveFailure(String resourceId, String sbrId, Throwable ex) {
        LOG.warn("Sandbox resources resolver received error for {} resolving '{}' - {}: {}",
                fullDeployUnitId, resourceId, sbrId, ex.getMessage());
        handleNewEvent();
    }

    private void updateDocker(UpdateReason reason) {
        Set<DockerImageDescription> newImages = new HashSet<>(currentSpec.getImagesForBoxes().values());
        Set<DockerImageDescription> addedImages = new HashSet<>(Sets.difference(newImages, dockerRequiredImages));
        Set<DockerImageDescription> removedImages = new HashSet<>(Sets.difference(dockerRequiredImages, newImages));

        if (reason == UpdateReason.SPEC_UPDATED) {
            forceResolvingForMissedDockerImageDigest(newImages);
        }

        LOG.debug("{}: checking required docker image contents: {} current -> {} target, {} new, {} removed",
                fullDeployUnitId, dockerRequiredImages.size(), newImages.size(), addedImages.size(), removedImages.size());

        removedImages.forEach(description -> dockerImagesResolver.removeResolve(description, fullDeployUnitId));
        addedImages.forEach(description -> dockerImagesResolver.addResolve(description, fullDeployUnitId));

        dockerRequiredImages = newImages;
    }

    private void forceResolvingForMissedDockerImageDigest(Set<DockerImageDescription> newImages) {
        newImages.forEach(description -> {
            DockerResolveStatus status = dockerImagesResolver.getResolveStatus(description);
            if (status.getReadiness().isReady()) {
                dockerImagesResolver.forceResolve(description);
            }
        });
    }

    private void updateSandbox() {
        sandboxResolver.registerResultHandlerAndTryGet(fullDeployUnitId, this, currentSpec,
                stageContext.getDynamicResources().values().stream()
                        .filter(spec -> spec.getDeployUnitRef().equals(deployUnitId))
                        .map(DynamicResourceSpec::getDynamicResource)
                        .collect(Collectors.toList())
        );
    }

    private void calcDockerResolverResult() {
        dockerResolveResult = dockerRequiredImages.stream().collect(Collectors.toMap(description -> description,
                description ->
                        dockerImagesResolver.getResolveStatus(description).getResult().orElseThrow()));
    }

    private boolean prerequisitesAreReady() {
        return calcDockerResolveStatus().isReady()
                && sandboxResolver.getResolveStatus(fullDeployUnitId).isReady();
    }

    private DeployUnitContext makeContext() {
        return new DeployUnitContext(
                stageContext,
                currentSpec,
                deployUnitId,
                fullDeployUnitId,
                dockerResolveResult,
                OptionalF.when(isLogsTransmittingEnabled(), () -> logbrokerTopicConfigResolver.get(getDeployUnitInfo())),
                sandboxResolver.get(fullDeployUnitId));
    }

    private THorizontalPodAutoscalerSpec constructHorizontalPodAutoscaleSpec(TReplicaSetScaleSpec replicaSetScaleSpec) {
        return THorizontalPodAutoscalerSpec.newBuilder()
                .setReplicaSet(replicaSetScaleSpec)
                .build();
    }

    private String resolveItype() {
        String itypeInSpec = currentSpec.getDetails().getPodSpec().getHostInfra().getMonitoring().getLabelsMap()
                .get(MonitoringPatcherUtils.ITYPE_KEY);
        return Objects.requireNonNullElse(itypeInSpec, MonitoringPatcherUtils.ITYPE_DEPLOY);
    }

    private String childNodeFullId(String id) {
        return fullDeployUnitId + StringUtils.ID_SEPARATOR + id;
    }
}
