package ru.yandex.infra.stage.deployunit;

import java.time.Clock;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.MoreObjects;

import ru.yandex.infra.stage.dto.Condition;

import static ru.yandex.infra.stage.deployunit.DeployUnitControllerImpl.DEFAULT_LATEST_DEPLOYED_REVISION;

public class DeployUnitTimeline {
    private final long targetRevision;
    private final Instant startTimestamp;
    private final Optional<Instant> finishTimestamp;
    private final Status status;
    private final Condition latestReadyCondition;
    private final long latestDeployedRevision;

    public DeployUnitTimeline(Clock clock) {
        this(0L, Instant.EPOCH,
                new Condition(Condition.Status.UNKNOWN, "", "", clock.instant()));
    }

    public DeployUnitTimeline(long targetRevision, Instant startTimestamp, Condition latestReadyCondition) {
        this(targetRevision, startTimestamp, Optional.empty(),
                Status.DEPLOYED, latestReadyCondition, DEFAULT_LATEST_DEPLOYED_REVISION);
    }

    public DeployUnitTimeline(Condition latestReadyCondition, long latestDeployedRevision) {
        this(0L, Instant.EPOCH, latestReadyCondition, latestDeployedRevision);
    }

    public DeployUnitTimeline(long targetRevision,
                              Instant startTimestamp,
                              Condition latestReadyCondition,
                              long latestDeployedRevision) {
        this(targetRevision, startTimestamp, Optional.empty(), Status.DEPLOYED, latestReadyCondition, latestDeployedRevision);
    }

    public DeployUnitTimeline(long targetRevision, Instant startTimestamp, Optional<Instant> finishTimestamp,
                              Status status, Condition latestReadyCondition, long latestDeployedRevision) {
        this.targetRevision = targetRevision;
        this.startTimestamp = startTimestamp;
        this.finishTimestamp = finishTimestamp;
        this.status = status;
        this.latestReadyCondition = latestReadyCondition;
        this.latestDeployedRevision = latestDeployedRevision;
    }

    public Optional<Long> getDeployingTime() {
        return finishTimestamp.map(finish -> finish.toEpochMilli() - startTimestamp.toEpochMilli());
    }

    public long getTargetRevision() {
        return targetRevision;
    }

    public Instant getStartTimestamp() {
        return startTimestamp;
    }

    public Optional<Instant> getFinishTimestamp() {
        return finishTimestamp;
    }

    public Status getStatus() {
        return status;
    }

    public Condition getLatestReadyCondition() {
        return latestReadyCondition;
    }

    public long getLatestDeployedRevision() {
        return latestDeployedRevision;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("targetRevision", targetRevision)
                .add("startTimestamp", startTimestamp)
                .add("finishTimestamp", finishTimestamp)
                .add("status", status)
                .add("latestReadyCondition", latestReadyCondition)
                .add("latestDeployedRevision", latestDeployedRevision)
                .toString();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof DeployUnitTimeline)) {
            return false;
        }
        DeployUnitTimeline that = (DeployUnitTimeline) obj;
        return  Objects.equals(targetRevision, that.targetRevision) &&
                Objects.equals(startTimestamp, that.startTimestamp) &&
                Objects.equals(finishTimestamp, that.finishTimestamp) &&
                Objects.equals(status, that.status) &&
                Objects.equals(latestReadyCondition, that.latestReadyCondition) &&
                Objects.equals(latestDeployedRevision, that.latestDeployedRevision);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                targetRevision,
                startTimestamp,
                finishTimestamp,
                status,
                latestReadyCondition,
                latestDeployedRevision
        );
    }

    public enum Status {
        DEPLOYING, DEPLOYED
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private long targetRevision;
        private Instant startTimestamp;
        private Optional<Instant> finishTimestamp;
        private Status status;
        private Condition latestReadyCondition;
        private long latestDeployedRevision;

        public Builder(DeployUnitTimeline deployUnitTimeline) {
            this.targetRevision = deployUnitTimeline.targetRevision;
            this.startTimestamp = deployUnitTimeline.startTimestamp;
            this.finishTimestamp = deployUnitTimeline.finishTimestamp;
            this.status = deployUnitTimeline.status;
            this.latestReadyCondition = deployUnitTimeline.latestReadyCondition;
            this.latestDeployedRevision = deployUnitTimeline.latestDeployedRevision;
        }

        public Builder withTargetRevision(long targetRevision) {
            this.targetRevision = targetRevision;
            return this;
        }

        public Builder withStartTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
            return this;
        }

        public Builder withFinishTimestamp(Optional<Instant> finishTimestamp) {
            this.finishTimestamp = finishTimestamp;
            return this;
        }

        public Builder withStatus(Status status) {
            this.status = status;
            return this;
        }

        public Builder withLatestReadyCondition(Condition latestReadyCondition) {
            this.latestReadyCondition = latestReadyCondition;
            return this;
        }

        public Builder withLatestDeployedRevision(long latestDeployedRevision) {
            this.latestDeployedRevision = latestDeployedRevision;
            return this;
        }

        public DeployUnitTimeline build() {
            return new DeployUnitTimeline(
                    targetRevision,
                    startTimestamp,
                    finishTimestamp,
                    status,
                    latestReadyCondition,
                    latestDeployedRevision
            );
        }
    }
}
