package ru.yandex.infra.stage.deployunit;

import java.time.Clock;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import com.codahale.metrics.MetricRegistry;
import io.opentracing.Span;
import io.opentracing.util.GlobalTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.dto.Condition;
import ru.yandex.infra.stage.dto.DeployUnitStatus;
import ru.yandex.qe.telemetry.metrics.Gauges;

public class DeployUnitTimelineManager {
    private static final Logger LOG = LoggerFactory.getLogger(DeployUnitTimeline.class);
    private static final String METRIC_PREFIX = "deploy_time_";

    private static Set<String> deployTimelineStages = Set.of();
    private static MetricRegistry metricRegistry;
    private static Map<String, Long> deployNameToDeployTime;

    public static void setupMetric(List<String> deployTimelineStages, MetricRegistry metricRegistry) {
        DeployUnitTimelineManager.deployTimelineStages = new HashSet<>(deployTimelineStages);
        DeployUnitTimelineManager.metricRegistry = metricRegistry;
        DeployUnitTimelineManager.deployNameToDeployTime = new ConcurrentHashMap<>();
    }

    private final Clock clock;
    private final String stageId;
    private final String fullDeployUnitId;
    private DeployUnitTimeline deployUnitTimeline;

    public DeployUnitTimelineManager(Clock clock, String stageId, String fullDeployUnitId) {
        this(clock, stageId, fullDeployUnitId, new DeployUnitTimeline(clock));
    }

    public DeployUnitTimelineManager(Clock clock, String stageId, String fullDeployUnitId, DeployUnitTimeline deployUnitTimeline) {
        this.clock = clock;
        this.stageId = stageId;
        this.fullDeployUnitId = fullDeployUnitId;
        this.deployUnitTimeline = deployUnitTimeline;
    }

    public void restoreFromStatus(DeployUnitStatus status) {
        deployUnitTimeline = status.getDeployUnitTimeline();
    }

    public void update(Readiness readiness, int currentSpecRevision) {
        Instant timestamp = readiness.isReady() == deployUnitTimeline.getLatestReadyCondition().isTrue() &&
                currentSpecRevision == deployUnitTimeline.getTargetRevision()
                ? deployUnitTimeline.getLatestReadyCondition().getTimestamp()
                : clock.instant();

        Condition latestReadyCondition = readiness.getReadyCondition(timestamp);
        long latestDeployedRevision = deployUnitTimeline.getLatestDeployedRevision();
        long prevLatestDeployedRevision = latestDeployedRevision;


        if (latestReadyCondition.isTrue()) {
            if (latestDeployedRevision < currentSpecRevision) {
                latestDeployedRevision = currentSpecRevision;
//                Deploy finish now detected
                Span span = GlobalTracer.get().buildSpan("du-deployed")
                        .withTag("stage_id", stageId)
                        .withTag("deploy_unit_id", fullDeployUnitId)
                        .withTag("revision", latestDeployedRevision)
                        .start();
                span.finish();
            } else if (latestDeployedRevision > currentSpecRevision) {
                LOG.warn("Deploy unit {}: latest deployed revision is {}, but current spec revision is {}",
                        fullDeployUnitId, latestDeployedRevision, currentSpecRevision);
            }
        }

        boolean isDeployed = latestDeployedRevision == currentSpecRevision;
        boolean wasDeployed = deployUnitTimeline.getStatus() == DeployUnitTimeline.Status.DEPLOYED;
        DeployUnitTimeline.Status currentStatus = isDeployed ? DeployUnitTimeline.Status.DEPLOYED : DeployUnitTimeline.Status.DEPLOYING;
        Instant startTimestamp = !isDeployed && (wasDeployed || deployUnitTimeline.getTargetRevision() != currentSpecRevision) ?
                timestamp :
                deployUnitTimeline.getStartTimestamp();

        Optional<Instant> finishTimestamp =
                isDeployed ?
                        wasDeployed ?
                                deployUnitTimeline.getFinishTimestamp()
                                : Optional.of(timestamp)
                        : Optional.empty();

        if (!isDeployed && !wasDeployed && deployUnitTimeline.getTargetRevision() != currentSpecRevision) {
            LOG.info("Deploying from {} to {} revision for {} was canceled in order to deploy {} revision",
                    deployUnitTimeline.getLatestDeployedRevision(), deployUnitTimeline.getTargetRevision(), fullDeployUnitId, currentSpecRevision);
        }

        if (!isDeployed && wasDeployed) {
            LOG.info("Start deploying from {} to {} revision for {}",
                    deployUnitTimeline.getLatestDeployedRevision(), currentSpecRevision, fullDeployUnitId);
        }

        deployUnitTimeline = new DeployUnitTimeline(
                currentSpecRevision,
                startTimestamp,
                finishTimestamp,
                currentStatus,
                latestReadyCondition,
                latestDeployedRevision
        );

        if (isDeployed && !wasDeployed) {
            updateMetric();
            LOG.info("Finish deploying from {} to {} revision for {} with deploying time = {} milliseconds",
                    prevLatestDeployedRevision, deployUnitTimeline.getLatestDeployedRevision(), fullDeployUnitId, deployUnitTimeline.getDeployingTime().get());
        }
    }

    private void updateMetric() {
        if (deployTimelineStages.contains(stageId) && deployUnitTimeline.getFinishTimestamp().isPresent()) {
            if (!deployNameToDeployTime.containsKey(fullDeployUnitId)) {
                Gauges.forSupplier(metricRegistry, METRIC_PREFIX + fullDeployUnitId,
                        () -> deployNameToDeployTime.getOrDefault(fullDeployUnitId, 0L));
            }
            deployNameToDeployTime.put(fullDeployUnitId, deployUnitTimeline.getDeployingTime().get());
        }
    }

    public DeployUnitTimeline getDeployUnitTimeline() {
        return deployUnitTimeline;
    }

    public Instant getTimestamp() {
        return deployUnitTimeline.getLatestReadyCondition().getTimestamp();
    }

    public static Set<String> getDeployTimelineStages() {
        return deployTimelineStages;
    }

    public static MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }
}
