package ru.yandex.infra.stage.deployunit;

import java.util.Optional;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.dto.DynamicResourceMeta;
import ru.yandex.infra.stage.dto.DynamicResourceRevisionStatus;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.SpecStatusMeta;
import ru.yandex.infra.stage.yp.SpecWithAcl;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceSpec;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceStatus;
import ru.yandex.yp.model.YpObjectType;

public class DynamicResourceController extends YpObjectControllerBase<DynamicResourceMeta, TDynamicResourceSpec,
        TDynamicResourceStatus, DynamicResourceRevisionStatus> {
    private static final Logger LOG = LoggerFactory.getLogger(DynamicResourceController.class);
    private static final String POD_SET_ID_META_FIELD = "pod_set_id";
    private final String podSetId;
    private final Converter converter;

    public DynamicResourceController(String id, ObjectLifeCycleManager<DynamicResourceMeta,
            TDynamicResourceSpec, TDynamicResourceStatus> repository,
                                     Consumer<DynamicResourceRevisionStatus> updateNotifier, String podSetId,
                                     Converter converter) {
        super(id, repository, updateNotifier, AclUpdater.IDENTITY, YpObjectType.DYNAMIC_RESOURCE);
        this.podSetId = podSetId;
        this.converter = converter;
    }

    @Override
    protected DynamicResourceRevisionStatus getStatusForNewSpec() {
        return DynamicResourceRevisionStatus.emptyStatus();
    }

    // Just ignore, handleCurrentState will setup right status for cluster
    @Override
    protected DynamicResourceRevisionStatus getStatus(Readiness readiness) {
        return DynamicResourceRevisionStatus.emptyStatus();
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        builder.addReadyDynamicResource(getStatus().getReady().getPodCount());
        builder.addUnreadyDynamicResource(getStatus().getInProgress().getPodCount() + getStatus().getError().getPodCount());
    }

    @Override
    protected void handleCurrentState(Optional<SpecStatusMeta<DynamicResourceMeta, TDynamicResourceSpec,
            TDynamicResourceStatus>> response) {
        if (currentSpecOpt.isPresent()) {
            SpecWithAcl<TDynamicResourceSpec> specWithAcl = currentSpecOpt.get();
            response.ifPresentOrElse(value -> {
                TDynamicResourceStatus fullStatus = value.getStatus();
                DynamicResourceRevisionStatus status = fullStatus.getRevisionsList().stream()
                        .reduce((s1, s2) -> s1.getRevision() > s2.getRevision() ? s1 : s2)
                        .map(converter::fromProto)
                        .orElse(DynamicResourceRevisionStatus.emptyStatus());
                if (!value.getSpec().equals(specWithAcl.getSpec())) {
                    LOG.info("Updating spec for dynamic resource {}", podSetId);
                    updateObjectSpec(specWithAcl.getSpec());
                } else if (!specWithAcl.getAcl().equals(value.getMeta().getAcl())) {
                    LOG.info("Updating acl for  dynamic resource  {}", podSetId);
                    updateObjectAcl(specWithAcl.getAcl());
                }
                updateStatus(status);
            }, () -> createObject(constructSimpleCreateRequestBuilder(specWithAcl)
                    .addSpecificMetaField(POD_SET_ID_META_FIELD, podSetId)
                    .build()));
        } else {
            LOG.debug("Spec is absent for dynamic resource controller {}, will ignore current state", podSetId);
        }
    }

    @Override
    protected Logger getLogger() {
        return LOG;
    }
}
