package ru.yandex.infra.stage.deployunit;

import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.SpecStatusMeta;
import ru.yandex.infra.stage.yp.SpecWithAcl;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.model.YpObjectType;

public class EndpointSetController extends YpObjectControllerBase<SchemaMeta, DataModel.TEndpointSetSpec, DataModel.TEndpointSetStatus, ReadinessStatus> {
    private static final Logger LOG = LoggerFactory.getLogger(EndpointSetController.class);

    //single-time adhoc solution for https://st.yandex-team.ru/DEPLOY-5903
    //should be removed after enabling all man endpoint_sets
    public static final String SUPERVISOR_LABEL_KEY = "supervisor";
    public static final String SUPERVISOR_LABEL_VALUE = "freezed-2022-04-24";

    private final String endpointSetId;

    public EndpointSetController(
            String endpointSetId,
            ObjectLifeCycleManager<SchemaMeta, DataModel.TEndpointSetSpec, DataModel.TEndpointSetStatus> repository,
            Consumer<ReadinessStatus> updateNotifier) {
        super(endpointSetId, repository, updateNotifier, AclUpdater.IDENTITY, YpObjectType.ENDPOINT_SET);
        this.endpointSetId = endpointSetId;
    }

    @Override
    protected ReadinessStatus getStatusForNewSpec() {
        return new ReadinessStatus(Readiness.inProgress("ENDPOINT_SET_OUT_OF_SYNC"));
    }

    @Override
    protected ReadinessStatus getStatus(Readiness readiness) {
        return new ReadinessStatus(readiness);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        if (getStatus().getReadiness().isReady()) {
            builder.addReadyEndpointSet();
        } else {
            builder.addUnreadyEndpointSet();
        }
    }

    @VisibleForTesting
    static boolean containsFreezeLabel(Map<String, YTreeNode> labels) {
        YTreeNode node = labels.get(SUPERVISOR_LABEL_KEY);
        return node != null && node.isStringNode() && node.stringValue().equals(SUPERVISOR_LABEL_VALUE);
    }

    @Override
    protected void handleCurrentState(
            Optional<SpecStatusMeta<SchemaMeta, DataModel.TEndpointSetSpec, DataModel.TEndpointSetStatus>> response) {
        if (currentSpecOpt.isPresent()) {
            SpecWithAcl<DataModel.TEndpointSetSpec> specWithAcl = currentSpecOpt.get();
            response.ifPresentOrElse(value -> {
                if (!value.getSpec().equals(specWithAcl.getSpec())) {
                    LOG.info("Updating spec for endpoint set {}", endpointSetId);
                    updateStatus(new ReadinessStatus(Readiness.inProgress("UPDATING_ENDPOINT_SET_SPEC")));
                    updateObjectSpec(specWithAcl.getSpec());
                } else if (!specWithAcl.getAcl().equals(value.getMeta().getAcl())) {
                    LOG.info("Updating acl for endpoint set {}", endpointSetId);
                    updateObjectAcl(specWithAcl.getAcl());
                } else if (containsFreezeLabel(value.getLabels())) { //crunch, should be removed after enabling all man endpoint_sets
                    LOG.info("Removing {} label for endpoint set {}", SUPERVISOR_LABEL_KEY, endpointSetId);
                    removeLabel(SUPERVISOR_LABEL_KEY);
                }
                else {
                    updateStatus(new ReadinessStatus(Readiness.ready()));
                }
            }, () -> createObject(constructSimpleCreateRequestBuilder(specWithAcl).build()));
        } else {
            LOG.debug("Spec is absent for endpoint set controller {}, will ignore current state", endpointSetId);
        }
    }

    @Override
    protected Logger getLogger() {
        return LOG;
    }
}
