package ru.yandex.infra.stage.deployunit;

import java.util.Optional;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.dto.HorizontalPodAutoscalerMeta;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.SpecStatusMeta;
import ru.yandex.infra.stage.yp.SpecWithAcl;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerSpec;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerStatus;
import ru.yandex.yp.model.YpObjectType;

public class HorizontalPodAutoscalerController
        extends YpObjectControllerBase<HorizontalPodAutoscalerMeta, THorizontalPodAutoscalerSpec, THorizontalPodAutoscalerStatus, ReadinessStatus> {
    private static final Logger LOG = LoggerFactory.getLogger(HorizontalPodAutoscalerController.class);
    private static final String REPLICA_SET_META_FIELD = "replica_set_id";
    private final String replicaSetId;
    private final String horizontalPodAutoscalerId;

    public HorizontalPodAutoscalerController(
            String replicaSetParentId,
            ObjectLifeCycleManager<HorizontalPodAutoscalerMeta, THorizontalPodAutoscalerSpec,
                    THorizontalPodAutoscalerStatus> repository,
            Consumer<ReadinessStatus> updateNotifier,
            AclUpdater aclUpdater) {
        super(replicaSetParentId, repository, updateNotifier, aclUpdater, YpObjectType.HORIZONTAL_POD_AUTOSCALER);
        this.replicaSetId = replicaSetParentId;
        this.horizontalPodAutoscalerId = replicaSetParentId;
    }

    @Override
    protected ReadinessStatus getStatusForNewSpec() {
        return new ReadinessStatus(Readiness.inProgress("HORIZONTAL_POD_AUTOSCALER_OUT_OF_SYNC"));
    }

    @Override
    protected ReadinessStatus getStatus(Readiness readiness) {
        return new ReadinessStatus(readiness);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        if (getStatus().getReadiness().isReady()) {
            builder.addReadyHorizontalPodAutoscaler();
        } else {
            builder.addUnreadyHorizontalPodAutoscaler();
        }
    }

    @Override
    protected void handleCurrentState(
            Optional<SpecStatusMeta<HorizontalPodAutoscalerMeta, THorizontalPodAutoscalerSpec, THorizontalPodAutoscalerStatus>> response) {
        if (currentSpecOpt.isPresent()) {
            SpecWithAcl<THorizontalPodAutoscalerSpec> specWithAcl = currentSpecOpt.get();
            response.ifPresentOrElse(value -> {
                if (!value.getSpec().equals(specWithAcl.getSpec())) {
                    LOG.info("Updating spec for horizontal pod autoscaler {}", horizontalPodAutoscalerId);
                    updateStatus(new ReadinessStatus(Readiness.inProgress("UPDATING_HORIZONTAL_POD_SPEC")));
                    updateObjectSpec(specWithAcl.getSpec());
                } else if (!specWithAcl.getAcl().equals(value.getMeta().getAcl())) {
                    LOG.info("Updating acl for horizontal pod autoscaler {}", horizontalPodAutoscalerId);
                    updateObjectAcl(specWithAcl.getAcl());
                } else {
                    updateStatus(new ReadinessStatus(Readiness.ready()));
                }
            }, () -> createObject(constructSimpleCreateRequestBuilder(specWithAcl)
                    .addSpecificMetaField(REPLICA_SET_META_FIELD, replicaSetId)
                    .build()));
        } else {
            LOG.debug("Spec is absent for horizontal pod autoscaler controller {}, will ignore current state", horizontalPodAutoscalerId);
        }
    }

    @Override
    protected Logger getLogger() {
        return LOG;
    }
}
