package ru.yandex.infra.stage.deployunit;

import java.util.Optional;
import java.util.function.Consumer;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.stage.dto.DeployProgress;
import ru.yandex.infra.stage.dto.DeploymentStrategy;
import ru.yandex.infra.stage.primitives.DeployPrimitiveStatus;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.RelationController;
import ru.yandex.yp.client.api.EConditionStatus;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetStatus;
import ru.yandex.yp.model.YpObjectType;

public class MultiClusterReplicaSetController
        extends ReplicaSetControllerBase<TMultiClusterReplicaSetSpec, TMultiClusterReplicaSetStatus, Long> {
    private final String description;
    private final Converter converter;

    public MultiClusterReplicaSetController(
            String replicaSetId,
            ObjectLifeCycleManager<SchemaMeta, TMultiClusterReplicaSetSpec, TMultiClusterReplicaSetStatus> repository,
            Consumer<DeployPrimitiveStatus<TMultiClusterReplicaSetStatus>> updateNotifier,
            Converter converter,
            AclUpdater aclUpdater,
            String stageFqid,
            RelationController relationController) {
        super(replicaSetId, repository, updateNotifier, aclUpdater, YpObjectType.MULTI_CLUSTER_REPLICA_SET, stageFqid, relationController);
        this.description = String.format("multi cluster replica set %s", replicaSetId);
        this.converter = converter;
    }

    @Override
    protected String selfDescription() {
        return description;
    }

    @Override
    protected Long extractSpecRevisionId(TMultiClusterReplicaSetSpec spec) {
        return spec.getRevision();
    }

    @Override
    protected Long extractStatusRevisionId(TMultiClusterReplicaSetStatus status) {
        return status.getRevision();
    }

    @Override
    protected DeployProgress extractDeployProgress(TMultiClusterReplicaSetStatus status) {
        return status.getMappedClusterDeployStatusesMap().values().stream()
                .map(clusterStatus -> converter.fromProto(clusterStatus.getDetails().getCurrentRevisionProgress()))
                .reduce(DeployProgress::sum).orElse(emptyProgress());
    }

    @Override
    protected int extractReplicaCount(TMultiClusterReplicaSetSpec spec) {
        return spec.getClustersList().stream()
                .mapToInt(perCluster -> perCluster.getSpec().getReplicaCount())
                .sum();
    }

    @Override
    protected boolean isReady(TMultiClusterReplicaSetStatus status) {
        return status.getReady().getStatus() == EConditionStatus.CS_TRUE;
    }

    @Override
    protected TMultiClusterReplicaSetSpec patchReplicaCount(TMultiClusterReplicaSetSpec spec, int replicaCount) {
        return spec;
    }

    @Override
    protected DeploymentStrategy extractDeploymentStrategy(TMultiClusterReplicaSetSpec spec) {
        return converter.fromProto(spec.getDeploymentStrategy());
    }

    @Override
    protected Optional<TMultiClusterReplicaSetStatus> emptyYpStatus() {
        return Optional.empty();
    }
}
