package ru.yandex.infra.stage.deployunit;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;

// Represent objects with same id in multiple clusters
public class MultiplexingController<Spec, Status extends ReadyStatus> {

    private static final ExecutorService executor = Executors.newSingleThreadExecutor();

    private final Factory<Spec, Status> perClusterControllerFactory;
    private final String id;
    private final Consumer<Status> updateNotifier;
    private final String description;

    private DeployController<Spec, Status> syncController;

    public MultiplexingController(String id, Factory<Spec, Status> perClusterControllerFactory, String description,
                                  Consumer<Status> updateNotifier) {
        this.perClusterControllerFactory = perClusterControllerFactory;
        this.id = id;
        this.description = description;
        this.updateNotifier = updateNotifier;
        syncController = new ParallelController<>(id, perClusterControllerFactory, description, updateNotifier);
    }

    @FunctionalInterface
    public interface Factory<Spec, Status> {
        ObjectController<Spec, Status> createController(String id, String cluster, Consumer<Status> updateNotifier);
    }

    public void syncSequential(Map<String, Spec> newSpecs,
                               StageContext stageContext,
                               Set<String> approvedSet,
                               Map<String, Map<String, YTreeNode>> labels,
                               List<Pair<String, Boolean>> clusterSequence) {

        Map<String, Spec> specs = filterByDisabledClusters(newSpecs, stageContext);
        List<Pair<String, Boolean>> sequence = filterByDisabledClusters(clusterSequence, stageContext);
        getSequentialController(specs, stageContext, labels, sequence)
                .sync(specs, stageContext, approvedSet, labels, sequence);
    }

    public void syncParallel(Map<String, Spec> newSpecs,
                             StageContext stageContext,
                             Set<String> approvedSet,
                             Map<String, Map<String, YTreeNode>> labels,
                             List<Pair<String, Boolean>> clusterSequence) {
        Map<String, Spec> specs = filterByDisabledClusters(newSpecs, stageContext);
        List<Pair<String, Boolean>> sequence = filterByDisabledClusters(clusterSequence, stageContext);
        getParallelController().sync(specs, stageContext, approvedSet, labels, sequence);
    }


    public void syncParallel(Map<String, Spec> newSpecs, StageContext stageContext) {
        syncParallel(newSpecs, stageContext, emptySet(), emptyMap(), emptyList());
    }

    private Map<String, Spec> filterByDisabledClusters(Map<String, Spec> newSpecs, StageContext stageContext) {
        if (stageContext.getDisabledClusters().stream().anyMatch(newSpecs::containsKey)) {
            return newSpecs.entrySet().stream().filter(entry -> !stageContext.getDisabledClusters().contains(entry.getKey()))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        }
        return newSpecs;
    }

    private List<Pair<String, Boolean>> filterByDisabledClusters(List<Pair<String, Boolean>> clusterSequence, StageContext stageContext) {
        return clusterSequence.stream()
                .filter(pair -> !stageContext.getDisabledClusters().contains(pair.getKey()))
                .collect(Collectors.toList());
    }

    public Map<String, Status> getClusterStatuses() {
        return syncController.getClusterStatuses();
    }

    DeployController.DeployControllerType getSyncControllerType() {
        return syncController.getType();
    }

    DeployController<Spec, Status> getSequentialController(Map<String, Spec> newSpecs,
                                                           StageContext stageContext,
                                                           Map<String, Map<String, YTreeNode>> labels,
                                                           List<Pair<String, Boolean>> clusterSequence) {
        if (syncController.getType().equals(DeployController.DeployControllerType.SEQUENTIAL)
                && syncController.isEquals(newSpecs, stageContext, labels, clusterSequence)) {
            return syncController;
        }
        syncController.shutdown();
        syncController = new SequentialController<>(id, perClusterControllerFactory, updateNotifier,
                executor);
        return syncController;
    }

    DeployController<Spec, Status> getParallelController() {
        if (syncController.getType().equals(DeployController.DeployControllerType.PARALLEL)) {
            return syncController;
        }
        syncController.shutdown();
        syncController = new ParallelController<>(id, perClusterControllerFactory, description, updateNotifier);
        return syncController;
    }

    public void shutdown() {
        syncController.shutdown();
    }

    public void addStats(DeployUnitStats.Builder builder) {
        syncController.addStats(builder);
    }

}
