package ru.yandex.infra.stage.deployunit;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.Collections.emptyMap;

public class ParallelController<Spec, Status extends ReadyStatus> extends DeployController<Spec, Status> {
    private static final Logger LOG = LoggerFactory.getLogger(ParallelController.class);
    private final MultiplexingController.Factory<Spec, Status> perClusterControllerFactory;
    private final String id;
    private final Consumer<Status> updateNotifier;
    private final String description;
    private final Map<String, ObjectController<Spec, Status>> currentControllers = new HashMap<>();


    public ParallelController(String id,
                              MultiplexingController.Factory<Spec, Status> perClusterControllerFactory,
                              String description,
                              Consumer<Status> updateNotifier) {
        this.perClusterControllerFactory = perClusterControllerFactory;
        this.id = id;
        this.description = description;
        this.updateNotifier = updateNotifier;
    }

    @Override
    public boolean isEquals(Map<String, Spec> newSpecs,
                            StageContext stageContext,
                            Map<String, Map<String, YTreeNode>> labels,
                            List<Pair<String, Boolean>> clusterSequence) {
        return true;
    }

    @Override
    public void sync(Map<String, Spec> newSpecs,
                     StageContext stageContext,
                     Set<String> approvedSet,
                     Map<String, Map<String, YTreeNode>> labels,
                     List<Pair<String, Boolean>> clusterSequence) {
        Set<String> newClusters = newSpecs.keySet();
        Set<String> currentClusters = new HashSet<>(currentControllers.keySet());

        Sets.difference(currentClusters, newClusters).forEach(cluster -> {
            LOG.info("{}: removing cluster {} for object '{}'", description, cluster, id);
            currentControllers.get(cluster).shutdown();
            currentControllers.remove(cluster);
            todoApprove.remove(cluster);
        });

        Map<String, Boolean> collectApproveSettings = clusterSequence.stream().collect(Collectors.toMap(Pair::getKey,
                Pair::getValue));

        Sets.difference(newClusters, currentClusters).forEach(cluster -> {
            LOG.info("{}: adding cluster {} for object '{}'", description, cluster, id);
            ObjectController<Spec, Status> newController = perClusterControllerFactory.createController(id, cluster,
                    updateNotifier);

            todoApprove.put(cluster,
                    Pair.of(() -> newController.sync(newSpecs.get(cluster), stageContext,
                            labels.getOrDefault(cluster, emptyMap()), cluster),
                            collectApproveSettings.getOrDefault(cluster,
                            false)));

            currentControllers.put(cluster, newController);
        });

        Sets.intersection(newClusters, currentClusters).forEach(cluster -> todoApprove.put(cluster,
                Pair.of(() -> currentControllers.get(cluster).sync(newSpecs.get(cluster), stageContext,
                        labels.getOrDefault(cluster, emptyMap()), cluster), collectApproveSettings.getOrDefault(cluster,
                        false))));

        doApprove(approvedSet);
    }

    @Override
    public Map<String, Status> getClusterStatuses() {
        return EntryStream.of(currentControllers)
                .mapValues(ObjectController::getStatus)
                .toMap();
    }

    @Override
    public void shutdown() {
        currentControllers.values().forEach(ObjectController::shutdown);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        currentControllers.forEach((cluster, controller) -> controller.addStats(builder));
    }

    @Override
    public DeployControllerType getType() {
        return DeployControllerType.PARALLEL;
    }

}
