package ru.yandex.infra.stage.deployunit;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.dto.Condition;

public class Readiness {
    private final Optional<ReadinessDetails> inProgressDetails;
    private final Optional<ReadinessDetails> failureDetails;

    private Readiness(Optional<ReadinessDetails> inProgressDetails, Optional<ReadinessDetails> failureDetails) {
        this.inProgressDetails = inProgressDetails;
        this.failureDetails = failureDetails;
    }

    public static Readiness ready() {
        return new Readiness(Optional.empty(), Optional.empty());
    }

    // helpers to avoid long creation code lines
    public static Readiness inProgress(String reason) {
        return new Readiness(Optional.of(new ReadinessDetails(reason)), Optional.empty());
    }

    public static Readiness inProgress(String reason, String message) {
        return new Readiness(Optional.of(new ReadinessDetails(reason, message)), Optional.empty());
    }

    public static Readiness failed(String reason, String message) {
        return new Readiness(Optional.of(new ReadinessDetails(reason, message)), Optional.of(new ReadinessDetails(reason, message)));
    }

    public static Readiness failed(ReadinessDetails inProgressDetails, ReadinessDetails failureDetails) {
        return new Readiness(Optional.of(inProgressDetails), Optional.of(failureDetails));
    }

    public boolean isReady() {
        return inProgressDetails.isEmpty() && failureDetails.isEmpty();
    }

    public Optional<ReadinessDetails> getInProgressDetails() {
        return inProgressDetails;
    }

    public Optional<ReadinessDetails> getFailureDetails() {
        return failureDetails;
    }

    // Merge readiness from similar objects across different clusters
    public static Readiness mergeRelated(Map<String, Readiness> perClusterReadiness) {
        Map<Boolean, List<Map.Entry<String, Readiness>>> byReady = perClusterReadiness.entrySet().stream()
                .collect(Collectors.partitioningBy(entry -> entry.getValue().isReady()));
        List<Map.Entry<String, Readiness>> unreadyList = byReady.get(false);
        if (unreadyList.isEmpty()) {
            return Readiness.ready();
        } else {
            Map<String, ReadinessDetails> failures = EntryStream.of(perClusterReadiness)
                    .flatMapValues(readiness -> readiness.getFailureDetails().stream()).toMap();
            ReadinessDetails inProgressMergedDetails = ReadinessDetails.merge(EntryStream.of(unreadyList.stream())
                    .mapValues(readiness -> readiness.getInProgressDetails().get())
                    .toMap()
            );
            if (failures.isEmpty()) {
                return Readiness.inProgress(inProgressMergedDetails.getReason());
            } else {
                return Readiness.failed(inProgressMergedDetails, ReadinessDetails.merge(failures));
            }
        }
    }

    // Merge readiness of unrelated objects
    // Return unready if any of arguments is unready, preferring first
    public static Readiness mergeUnrelated(Readiness first, Readiness second) {
        return first.isReady() ? second : first;
    }

    public static Readiness mergeUnrelated(Readiness first, Readiness second, Readiness third) {
        return first.isReady() ? mergeUnrelated(second, third) : first;
    }

    //Merge readiness for more than 2 unrelated objects
    public static Readiness mergeUnrelated(Readiness... toMerge)
    {
        return Arrays.stream(toMerge).reduce(Readiness::mergeUnrelated).get();
    }

    public Condition getReadyCondition(Instant timestamp) {
        return getCondition(timestamp, Condition.Status.TRUE, Condition.Status.FALSE);
    }

    public Condition getInProgressCondition(Instant timestamp) {
        return getCondition(timestamp, Condition.Status.FALSE, Condition.Status.TRUE);
    }

    public Condition getFailureCondition(Instant timestamp) {
        if (failureDetails.isEmpty()) {
            return new Condition(Condition.Status.FALSE, "", "", timestamp);
        } else {
            ReadinessDetails details = failureDetails.get();
            return new Condition(Condition.Status.TRUE, details.getReason(), details.getMessage(), timestamp);
        }
    }

    private Condition getCondition(Instant timestamp, Condition.Status ifReady, Condition.Status ifUnready) {
        if (isReady()) {
            return new Condition(ifReady, "", "", timestamp);
        } else {
            ReadinessDetails details = inProgressDetails.get();
            return new Condition(ifUnready, details.getReason(), details.getMessage(), timestamp);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Readiness)) return false;
        Readiness readiness = (Readiness) o;
        return Objects.equals(inProgressDetails, readiness.inProgressDetails) &&
                Objects.equals(failureDetails, readiness.failureDetails);
    }

    @Override
    public int hashCode() {
        return Objects.hash(inProgressDetails, failureDetails);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("inProgressDetails", inProgressDetails)
                .add("failureDetails", failureDetails)
                .toString();
    }
}
