package ru.yandex.infra.stage.deployunit;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;

// Expected to be mapped to Condition.reason and Condition.message
public class ReadinessDetails {
    private final String reason;
    private final String message;

    public ReadinessDetails(String reason, String message) {
        this.reason = reason;
        this.message = message;
    }

    public ReadinessDetails(String reason) {
        this(reason, "");
    }

    public String getReason() {
        return reason;
    }

    public String getMessage() {
        return message;
    }

    public static ReadinessDetails merge(Map<String, ReadinessDetails> perClusterDetails) {
        if (perClusterDetails.isEmpty()) {
            throw new IllegalArgumentException("Cannot merge an empty map of descriptions");
        }
        Set<String> reasons = perClusterDetails.values().stream()
                .map(ReadinessDetails::getReason)
                .collect(Collectors.toSet());
        String joinedMessages = perClusterDetails.entrySet().stream()
                .filter(entry -> !entry.getValue().getMessage().isEmpty())
                .map(entry -> String.format("Cluster '%s': %s", entry.getKey(), entry.getValue().getMessage()))
                .collect(Collectors.joining("; "));
        if (reasons.size() == 1) {
            return new ReadinessDetails(reasons.iterator().next(), joinedMessages);
        } else {
            return new ReadinessDetails("MULTIPLE_REASONS", joinedMessages);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ReadinessDetails)) return false;
        ReadinessDetails that = (ReadinessDetails) o;
        return Objects.equals(reason, that.reason) &&
                Objects.equals(message, that.message);
    }

    @Override
    public int hashCode() {
        return Objects.hash(reason, message);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("reason", reason)
                .add("message", message)
                .toString();
    }
}
