package ru.yandex.infra.stage.deployunit;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.yp.CreateObjectRequest;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.infra.stage.dto.DeployProgress;
import ru.yandex.infra.stage.dto.DeploymentStrategy;
import ru.yandex.infra.stage.primitives.DeployPrimitiveStatus;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.RelationController;
import ru.yandex.infra.stage.yp.SpecStatusMeta;
import ru.yandex.infra.stage.yp.SpecWithAcl;
import ru.yandex.yp.model.YpObjectType;

import static java.util.Optional.empty;
import static ru.yandex.infra.stage.primitives.DeployPrimitiveController.DEPLOY_LABEL_KEY;
import static ru.yandex.infra.stage.primitives.ReplicaSetDeployPrimitiveController.AUTOSCALER_ID_LABEL_KEY;

public abstract class ReplicaSetControllerBase<YpSpec extends Message, YpStatus extends Message, RevisionType>
        extends YpObjectControllerBase<SchemaMeta, YpSpec, YpStatus, DeployPrimitiveStatus<YpStatus>> {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicaSetControllerBase.class);

    private final RelationController relationController;
    private final String initialStageFqid;
    private String replicaSetFqid;
    private int handleCurrentStateCounter = 0;

    ReplicaSetControllerBase(String replicaSetId,
                             ObjectLifeCycleManager<SchemaMeta, YpSpec, YpStatus> repository,
                             Consumer<DeployPrimitiveStatus<YpStatus>> updateNotifier,
                             AclUpdater aclUpdater,
                             YpObjectType type,
                             String stageFqid,
                             RelationController relationController) {
        super(replicaSetId, repository, updateNotifier, aclUpdater, type);
        this.initialStageFqid = stageFqid;
        this.relationController = relationController;
    }

    protected DeployProgress emptyProgress() {
        return new DeployProgress(0, 0, currentSpecOpt.map(value -> extractReplicaCount(value.getSpec())).orElse(0));
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        if (getStatus().getReadiness().isReady()) {
            builder.addReadyReplicaSet();
        } else {
            builder.addUnreadyReplicaSet();
        }
    }

    protected abstract String selfDescription();

    // TODO: replace with long
    protected abstract RevisionType extractSpecRevisionId(YpSpec spec);

    protected abstract RevisionType extractStatusRevisionId(YpStatus status);

    protected abstract DeployProgress extractDeployProgress(YpStatus status);

    protected abstract int extractReplicaCount(YpSpec spec);

    protected abstract boolean isReady(YpStatus status);

    protected abstract YpSpec patchReplicaCount(YpSpec spec, int replicaCount);

    protected abstract DeploymentStrategy extractDeploymentStrategy(YpSpec spec);

    protected abstract Optional<YpStatus> emptyYpStatus();

    @VisibleForTesting
    int getHandleCurrentStateCounter() {
        return handleCurrentStateCounter;
    }

    @Override
    protected DeployPrimitiveStatus<YpStatus> getStatusForNewSpec() {
        return new DeployPrimitiveStatus<>(Readiness.inProgress("REPLICA_SET_OUT_OF_SYNC"), emptyProgress(), emptyYpStatus());
    }

    @Override
    protected DeployPrimitiveStatus<YpStatus> getStatus(Readiness readiness) {
        return new DeployPrimitiveStatus<>(readiness, getStatus().getProgress(), emptyYpStatus());
    }

    @Override
    protected void handleCurrentState(Optional<SpecStatusMeta<SchemaMeta, YpSpec, YpStatus>> response) {
        if (currentSpecOpt.isPresent()) {
            handleCurrentStateCounter++;
            SpecWithAcl<YpSpec> currentThisSpec = currentSpecOpt.get();
            response.ifPresentOrElse(value -> syncExistedObject(currentThisSpec, value),
                    () -> createNewObject(currentThisSpec));
        } else {
            LOG.debug("Spec is absent for {}, will ignore current state", selfDescription());
        }
    }

    @Override
    protected void handleRemove() {
        if (replicaSetFqid != null) {
            try {
                relationController.removeRelation(getStageFqid(), replicaSetFqid).get();
            } catch (ExecutionException|InterruptedException ignored) {
                //already logged inside relationController, just skipping...
            }
            replicaSetFqid = null;
        }
    }

    private void createNewObject(SpecWithAcl<YpSpec> currentThisSpec) {
        final CreateObjectRequest<YpSpec> request = constructSimpleCreateRequestBuilder(currentThisSpec).build();

        replicaSetFqid = null;
        String stageFqid = getStageFqid();
        createObject(ypRepository -> {
            YpTransactionClient ypTransactionClient = ypRepository.getTransactionClient();
            return ypTransactionClient.runWithTransaction(transaction -> ypRepository.createObject(getReplicaSetId(), transaction, request)
                    .thenCompose(ypTypeId -> {
                        replicaSetFqid = ypTypeId.getFqid().orElseThrow();
                        return relationController.addRelation(stageFqid, replicaSetFqid);
                    })
            )
            .whenComplete((x, error) -> {
                if (error != null && replicaSetFqid != null) {
                    relationController.removeRelation(stageFqid, replicaSetFqid);
                }
            });
        });
    }

    private boolean checkRelationExists(SpecStatusMeta<SchemaMeta, YpSpec, YpStatus> metaSpecStatus) {
        final String fqid = metaSpecStatus.getMeta().getFqid();

        String stageFqid = getStageFqid();
        if (!relationController.containsRelation(stageFqid, fqid)) {
            var message = String.format("Found RS/MCRS '%s' with missed relation to stage '%s'. " +
                    "RS/MCRS should not exist without underlying record in 'relation' table. " +
                    "It occurs when stage is recreated with the same name, but GC was not able to remove old RS/MCRS on time. " +
                    "Please wait GC to complete it's job or ask support engineer to remove RS/MCRS manually.", fqid, stageFqid);
            LOG.warn(message);
            updateStatus(new DeployPrimitiveStatus<>(Readiness.failed("NO_RELATION_FOR_REPLICA_SET", message), emptyProgress(),
                    emptyYpStatus()));
            replicaSetFqid = null;
            return false;
        }

        replicaSetFqid = fqid;
        return true;
    }

    private void syncExistedObject(SpecWithAcl<YpSpec> currentThisSpec, SpecStatusMeta<SchemaMeta, YpSpec, YpStatus> metaSpecStatus) {

        if (!checkRelationExists(metaSpecStatus)) {
            return;
        }

        YpSpec ypSpec = metaSpecStatus.getSpec();
        YpStatus ypStatus = metaSpecStatus.getStatus();
        YpSpec targetSpec = currentThisSpec.getSpec();
        RevisionType targetRevisionId = extractSpecRevisionId(targetSpec);
        RevisionType specRevisionId = extractSpecRevisionId(ypSpec);
        DeploymentStrategy targetDeploymentStrategy = extractDeploymentStrategy(targetSpec);
        DeploymentStrategy specDeploymentStrategy = extractDeploymentStrategy(ypSpec);

        Optional<Long> timestampPrerequisite = empty();
        if (currentLabels.containsKey(DEPLOY_LABEL_KEY) && currentLabels.get(DEPLOY_LABEL_KEY).isMapNode()) {
            Optional<String> autoscalerId = currentLabels.get(DEPLOY_LABEL_KEY).mapNode()
                    .getStringO(AUTOSCALER_ID_LABEL_KEY);
            if (autoscalerId.isPresent() && !autoscalerId.get().isEmpty()) {
                targetSpec = patchReplicaCount(targetSpec, extractReplicaCount(ypSpec));
                timestampPrerequisite = Optional.of(metaSpecStatus.getSpecTimestamp());
            }
        }

        if (!targetRevisionId.equals(specRevisionId) ||
                !targetDeploymentStrategy.equals(specDeploymentStrategy) ||
                !Objects.equals(currentLabels.get(DEPLOY_LABEL_KEY), metaSpecStatus.getLabels().get(DEPLOY_LABEL_KEY))) {
            LOG.info("{} needs updating: target revision {}, current revision {}", selfDescription(), targetRevisionId, specRevisionId);
            updateObjectSpecWithLabels(targetSpec, currentLabels, timestampPrerequisite);
        } else if (!metaSpecStatus.getMeta().getAcl().equals(currentThisSpec.getAcl())) {
            LOG.info("{} acl needs updating", selfDescription());
            updateObjectAcl(currentThisSpec.getAcl());
        } else {
            if (extractStatusRevisionId(ypStatus).equals(specRevisionId)) {
                DeployProgress currentProgress = extractDeployProgress(ypStatus);
                if (isReady(ypStatus)) {
                    // everything is ok, report it
                    updateStatus(new DeployPrimitiveStatus<>(Readiness.ready(), currentProgress, Optional.of(ypStatus)));
                } else {
                    updateStatus(new DeployPrimitiveStatus<>(Readiness.inProgress("REPLICA_SET_UPDATING_PODS"), currentProgress,
                            Optional.of(ypStatus)));
                }
            } else {
                updateStatus(new DeployPrimitiveStatus<>(Readiness.inProgress("REPLICA_SET_UPDATING_PODS"), emptyProgress(),
                        Optional.of(ypStatus)));
            }
        }
    }

    @Override
    protected Logger getLogger() {
        return LOG;
    }

    private String getStageFqid() {
        return currentStageContext != null ? currentStageContext.getStageFqid() : initialStageFqid;
    }
}
