package ru.yandex.infra.stage.deployunit;


import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.dto.Checksum;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.util.JsonHttpGetter;

public class SandboxResourcesGetterImpl implements SandboxResourcesGetter {
    private final String token;
    private final String templateUrl;
    private final Cache<DownloadableResource> cache;
    private final boolean enableResourceCache;
    private final JsonHttpGetter jsonHttpGetter;

    public SandboxResourcesGetterImpl(String token,
                                      String templateUrl,
                                      Cache<DownloadableResource> cache,
                                      boolean enableResourceCache,
                                      JsonHttpGetter jsonHttpGetter) {
        this.token = token;
        this.templateUrl = templateUrl;
        this.cache = cache;
        this.enableResourceCache = enableResourceCache;
        this.jsonHttpGetter = jsonHttpGetter;
    }

    @Override
    public CompletableFuture<String> get(String resourceId) {
        if (this.enableResourceCache) {
            Optional<DownloadableResource> cachedResource = cache.get(resourceId);
            if (cachedResource.isPresent()) {
                return CompletableFuture.completedFuture(cachedResource.get().getUrl());
            }
        }

        return jsonHttpGetter.get(String.format(templateUrl, resourceId), token)
                .thenApply(root -> parseSandboxResponse(resourceId, root));
    }

    private String parseSandboxResponse(String resourceId, JsonNode node) {
        JsonNode rbtorrent = node.get("skynet_id");
        if (rbtorrent == null || rbtorrent.isNull()) {
            throw new RuntimeException("Sandbox resource has an empty skynet_id");
        }
        Checksum checksum = Optional.ofNullable(node.get("digest"))
                .filter(JsonNode::isTextual)
                .map(val -> Checksum.fromString(val.textValue()))
                .orElse(Checksum.EMPTY);
        String resourceUrl = rbtorrent.textValue();
        if (this.enableResourceCache) {
            cache.put(resourceId, new DownloadableResource(resourceUrl, checksum));
        }
        return resourceUrl;
    }

}
