package ru.yandex.infra.stage.deployunit;

import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.util.ExceptionUtils;
import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.util.AdaptiveRateLimiter;
import ru.yandex.infra.stage.util.GeneralUtils;
import ru.yandex.infra.stage.util.StoppableBackOff;
import ru.yandex.infra.stage.util.StoppableBackOffExecution;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceSpec;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TResourceGang;

public class SandboxResourcesResolverImpl implements SandboxResourcesResolver {
    private static final Logger LOG = LoggerFactory.getLogger(SandboxResourcesResolverImpl.class);

    private static final String GETTING_STATUS = "RESOLVING_SANDBOX_RESOURCES";
    private static final String GETTING_ERROR_STATUS = "RESOLVING_SANDBOX_RESOURCES_ERROR";

    private final HashMap<String, Map<String, String>> resolvedResources = new HashMap<>();
    private final HashMap<String, Map<String, Throwable>> lastErrorByDeployUnitId = new HashMap<>();
    private final SerialExecutor executor;
    private final StoppableBackOff timeoutFactory;
    private final SandboxResourcesGetter sandboxResourcesGetter;
    private final AdaptiveRateLimiter rateLimiter;
    private final Cache<DownloadableResource> cache;

    private final Map<String, SandboxResourcesResolveResultHandler> resultResolveHandlers = new HashMap<>();

    private final Map<String, CompletableFuture<String>> inProgressRequests = new ConcurrentHashMap<>();

    private final Map<String, Set<String>> deployUnitIdToResourceSet = new ConcurrentHashMap<>();
    private final Map<String, Set<String>> resourceToSubscribers = new ConcurrentHashMap<>();

    public SandboxResourcesResolverImpl(SerialExecutor executor,
                                        SandboxResourcesGetter sandboxResourcesGetter,
                                        Cache<DownloadableResource> cache,
                                        Duration initialRetryTimeout,
                                        Duration maxRetryTimeout,
                                        AdaptiveRateLimiter rateLimiter) {
        this.executor = executor;
        this.timeoutFactory = GeneralUtils.getTimeoutFactory(
                initialRetryTimeout.toMillis(), maxRetryTimeout.toMillis()
        );
        this.sandboxResourcesGetter = sandboxResourcesGetter;
        this.cache = cache;
        this.rateLimiter = rateLimiter;
    }

    @Override
    public Map<String, String> get(String fullDeployUnitId) {
        return resolvedResources.get(fullDeployUnitId);
    }

    @Override
    public void registerResultHandlerAndTryGet(String unitId, SandboxResourcesResolveResultHandler handler,
                                               DeployUnitSpec spec, List<TDynamicResourceSpec> dynamicResourceSpecs) {
        resultResolveHandlers.put(unitId, handler);

        tryGet(unitId, spec, dynamicResourceSpecs);
    }

    @Override
    public void unregisterResultHandler(String unitId) {
        resultResolveHandlers.remove(unitId);
    }

    @Override
    public Readiness getResolveStatus(String fullDeployUnitId) {

        for (Map.Entry<String, String> sbrToRb : resolvedResources.get(fullDeployUnitId).entrySet()) {
            String resourceId = sbrToRb.getKey();
            if (sbrToRb.getValue() == null) {
                Throwable lastError = lastErrorByDeployUnitId.get(fullDeployUnitId).get(resourceId);
                if (lastError == null) {
                    return Readiness.inProgress(GETTING_STATUS);
                }
                //TODO: throw multiple errors in one?
                else {
                    return Readiness.failed(GETTING_ERROR_STATUS,
                            String.format("Could not resolve '%s': %s",
                                    resourceId,
                                    lastError.getMessage()));
                }
            }
        }
        return Readiness.ready();
    }

    public void tryGet(String fullDeployUnitId,
                       DeployUnitSpec spec,
                       List<TDynamicResourceSpec> dynamicResourceSpecs) {
        TResourceGang resourceGang = spec.getDetails().getPodSpec().getPodAgentPayload().getSpec().getResources();
        List<ResourceInfo> resourceInfoList =
                StreamEx.of(resourceGang.getStaticResourcesList().stream().map(ResourceInfo::new))
                        .append(resourceGang.getLayersList().stream().map(ResourceInfo::new))
                        .append(spec.getBoxJugglerConfigs().values().stream()
                                .flatMap(c -> c.getArchivedChecks().stream())
                                .map(DownloadableResource::getUrl)
                                .map(ResourceInfo::new)
                        )
                        .append(dynamicResourceSpecs.stream()
                                .flatMap(drSpec -> drSpec.getDeployGroupsList().stream())
                                .flatMap(dg -> dg.getUrlsList().stream())
                                .map(ResourceInfo::new)
                        ).filter(resource -> ResourceInfo.isSbrUrl(resource.url))
                        .collect(Collectors.toList());

        resolvedResources.put(fullDeployUnitId, new HashMap<>());
        lastErrorByDeployUnitId.put(fullDeployUnitId, new HashMap<>());

        Set<String> oldResources = deployUnitIdToResourceSet.getOrDefault(fullDeployUnitId, Set.of());
        Set<String> newResources = resourceInfoList.stream().map(ResourceInfo::getId).collect(Collectors.toSet());
        deployUnitIdToResourceSet.put(fullDeployUnitId, newResources);

        Set<String> addedResources = Sets.difference(newResources, oldResources);
        Set<String> deletedResources = Sets.difference(oldResources, newResources);
        addedResources.forEach(resourceId -> resourceToSubscribers.compute(resourceId, (k, v) -> {
            Set<String> subscribers = v != null ? v : new HashSet<>();
            subscribers.add(fullDeployUnitId);
            return subscribers;
        }));
        deletedResources.forEach(resourceId -> resourceToSubscribers.computeIfPresent(resourceId, (k, v) -> {
            v.remove(fullDeployUnitId);
            return v;
        }));

        resourceInfoList
                .forEach(sbrResource -> {
                    resolvedResources.get(fullDeployUnitId).put(sbrResource.getId(), null);
                    cache.get(sbrResource.getSandboxId()).map(DownloadableResource::getUrl)
                            .ifPresentOrElse(url -> addResult(fullDeployUnitId, sbrResource, url, false),
                                    () -> resolveSandboxResource(sbrResource, fullDeployUnitId));
                });
    }

    private void resolveSandboxResource(ResourceInfo resource, String fullDeployUnitId) {
        if (!rateLimiter.tryAcquire()) {
            executor.schedule(() -> resolveSandboxResource(resource, fullDeployUnitId), Duration.ofSeconds(1));
            return;
        }

        StoppableBackOffExecution timeout = timeoutFactory.startStoppable();
        executor.executeOrRetry(
                () -> inProgressRequests.compute(resource.getSandboxId(), (sandboxId, httpRequestFuture) -> {
                    if (httpRequestFuture == null || httpRequestFuture.isCompletedExceptionally()) {
                        rateLimiter.incrementAndGet();
                        LOG.info("Resolving sandbox resource {} for deploy unit '{}'", sandboxId, fullDeployUnitId);
                        return sandboxResourcesGetter.get(sandboxId)
                                .whenComplete((x, error) -> rateLimiter.decrementAndGet());
                    }
                    return httpRequestFuture;
                }),
                result -> addResult(fullDeployUnitId, resource, result, true),
                throwable -> {
                    addError(fullDeployUnitId, resource, throwable, true);
                    if (!resultResolveHandlers.containsKey(fullDeployUnitId) ||
                            resourceToSubscribers.getOrDefault(resource.getId(), Set.of()).size() == 0) {
                        // SandboxResourceResolver::unregisterCallback was called from DeployUnitControllerImpl
                        timeout.stop();
                    }
                },
                timeout
        );
    }

    private void addError(String fullDeployUnitId, ResourceInfo resource, Throwable throwable, boolean executeCallback) {
        throwable = ExceptionUtils.stripCompletionException(throwable);
        lastErrorByDeployUnitId.computeIfAbsent(fullDeployUnitId, key -> new HashMap<>())
                .put(resource.getId(), throwable);
        if (executeCallback) {
            var handler = resultResolveHandlers.get(fullDeployUnitId);
            if (handler != null) {
                handler.onSandboxResourceResolveFailure(resource.getId(), resource.getSandboxId(), throwable);
            }
        }
    }

    private void addResult(String fullDeployUnitId, ResourceInfo resource, String result, boolean executeCallback) {
        try {
            resource.validateResource(result);
        } catch (Throwable throwable) {
            addError(fullDeployUnitId, resource, throwable, executeCallback);
            return;
        }
        resolvedResources.computeIfAbsent(fullDeployUnitId, id -> new HashMap<>()).put(resource.getId(), result);
        if (executeCallback) {
            var handler = resultResolveHandlers.get(fullDeployUnitId);
            if (handler != null) {
                handler.onSandboxResourceResolveSuccess();
            }
        }
    }

    private static class ResourceInfo {
        private static final String RB_TORRENT_PREFIX = "rbtorrent:";
        private static final String SBR_PREFIX = "sbr:";
        private final String url;
        private final String id;

        public static boolean isSbrUrl(String url) {
            return url.startsWith(SBR_PREFIX);
        }

        public String getSandboxId() {
            if (isSbrUrl(url)) {
                return url.substring(ResourceInfo.SBR_PREFIX.length());
            }
            throw new RuntimeException("Not valid Sandbox resource: " + url);
        }

        public String getId() {
            return id;
        }

        public String getUrl() {
            return url;
        }

        public ResourceInfo(TResource resource) {
            this.url = resource.getUrl();
            this.id = resource.getId();
        }

        public ResourceInfo(String url) {
            this.url = url;
            this.id = url;
        }

        public ResourceInfo(TLayer layer) {
            this.url = layer.getUrl();
            this.id = layer.getId();
        }

        public void validateResource(String resolved) {
            if (!resolved.startsWith(RB_TORRENT_PREFIX)) {
                throw new RuntimeException(String.format("Resolved url %s for resource %s is invalid", resolved,
                        getId()));
            }
        }
    }
}
