package ru.yandex.infra.stage.deployunit;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.function.Consumer;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.Collections.emptyMap;

public class SequentialController<Spec, Status extends ReadyStatus> extends DeployController<Spec, Status> {

    private static final Logger LOG = LoggerFactory.getLogger(SequentialController.class);
    private final MultiplexingController.Factory<Spec, Status> perClusterControllerFactory;
    private final String id;
    private final Consumer<Status> updateNotifier;
    private final Map<String, ObjectController<Spec, Status>> currentControllers = new ConcurrentHashMap<>();
    private final ExecutorService executor;
    private CurrentStep currentStep;
    private Set<String> approvedSet;
    private final Set<Future<?>> tasks = Collections.newSetFromMap(new ConcurrentHashMap<>());
    private volatile boolean shutdown = false;

    public SequentialController(String id, MultiplexingController.Factory<Spec, Status> perClusterControllerFactory,
                                Consumer<Status> updateNotifier, ExecutorService executor) {
        this.perClusterControllerFactory = perClusterControllerFactory;
        this.executor = executor;
        this.id = id;
        this.updateNotifier = updateNotifier;
    }

    @Override
    public void sync(Map<String, Spec> newSpecs, StageContext stageContext, Set<String> approvedSet,
                     Map<String, Map<String, YTreeNode>> labels,
                     List<Pair<String, Boolean>> clusterSequence) {
        addFutureAndClean(executor.submit(errorHandler(() -> {
            checkAndInit(newSpecs, stageContext, labels, clusterSequence);
            this.approvedSet = approvedSet;
            Pair<String, Boolean> current = currentStep.current();
            if (current != null) {
                todoApprove.put(current.getKey(),
                        Pair.of(() -> {
                                    LOG.info("Submit {} for {}", current.getKey(), id);
                                    createOrGetController(current.getKey()).sync(newSpecs.get(current.getKey()), stageContext,
                                            labels.getOrDefault(current.getKey(), emptyMap()), current.getKey());
                                },
                                current.getValue()));
            }
            doApprove(approvedSet);
        })));
    }

    private Runnable errorHandler(Runnable runnable) {
        return () -> {
            try {
                if (!shutdown) {
                    runnable.run();
                }
            } catch (Exception exception) {
                LOG.error("Unrecognized state for {}", id, exception);
                throw exception;
            }
        };
    }

    private void checkAndInit(Map<String, Spec> newSpecs, StageContext stageContext, Map<String, Map<String, YTreeNode>> labels,
                              List<Pair<String, Boolean>> clusterSequence) {
        if (currentStep == null) {
            currentStep = new CurrentStep(newSpecs, stageContext, labels, clusterSequence);
        }
        if (!isEquals(newSpecs, stageContext, labels, clusterSequence)) {
            throw new IllegalStateException("Unrecognized state. Sequential controller has initialed with other data.");
        }
        // Should create but not start controllers in all locations
        newSpecs.keySet().forEach(this::createOrGetController);
    }


    private ObjectController<Spec, Status> createOrGetController(String cluster) {
        return currentControllers.computeIfAbsent(cluster, (v) -> perClusterControllerFactory.createController(id,
                cluster,
                this::applyNext));
    }

    @VisibleForTesting
    void applyNext(Status status) {
        addFutureAndClean(executor.submit(errorHandler(() -> {
            Pair<String, Boolean> currentCluster = currentStep.current();
            LOG.debug("{} {} state is ready {}", id, currentCluster, status.isReady());
//        You need check the real cluster status, because controller (RSC) could apply the READY status twice.
//        It happens first time when rsc have instances in active = instances count - disruption budget
//        It happens second time when rsc have all instances in active
//        See more DEPLOY-3794
            if (status.isReady() &&
                    currentStep.hasNext() && currentCluster != null && currentControllers.get(currentCluster.getKey()).getStatus().isReady()) {
                Pair<String, Boolean> nextCluster = currentStep.next();
                LOG.info("State is ready for {}, cluster {}. Scheduled next {}", id, currentCluster.getKey(),
                        nextCluster.getKey());
                todoApprove.put(nextCluster.getKey(),
                        Pair.of(() -> {
                                    LOG.info("Submit {} for {}", nextCluster.getKey(), id);
                                    createOrGetController(nextCluster.getKey()).sync(currentStep.newSpecs.get(nextCluster.getKey()),
                                            currentStep.stageContext,
                                            currentStep.labels.getOrDefault(nextCluster.getKey(), emptyMap()),
                                            nextCluster.getKey());
                                },
                                nextCluster.getValue()));
                doApprove(approvedSet);
            }
            updateNotifier.accept(status);
        })));
    }

    private void addFutureAndClean(Future<?> future) {
        tasks.removeIf(Future::isDone);
        tasks.add(future);
    }

    @Override
    public Map<String, Status> getClusterStatuses() {
        return EntryStream.of(currentControllers)
                .mapValues(ObjectController::getStatus)
                .toMap();
    }

    @Override
    public void shutdown() {
        shutdown = true;
        tasks.forEach(v -> v.cancel(true));
        currentControllers.values().forEach(ObjectController::shutdown);
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        currentControllers.forEach((cluster, controller) -> controller.addStats(builder));
    }

    @Override
    public DeployControllerType getType() {
        return DeployControllerType.SEQUENTIAL;
    }

    @Override
    public boolean isEquals(Map<String, Spec> newSpecs,
                            StageContext stageContext,
                            Map<String, Map<String, YTreeNode>> labels,
                            List<Pair<String, Boolean>> clusterSequence) {
        return currentStep == null || currentStep.equals(new CurrentStep(newSpecs, stageContext, labels, clusterSequence));
    }

    private class CurrentStep {
        private final Map<String, Spec> newSpecs;
        private final StageContext stageContext;
        private final Map<String, Map<String, YTreeNode>> labels;
        private final List<Pair<String, Boolean>> clusterSequence;
        private int currentIndex = 0;

        private CurrentStep(Map<String, Spec> newSpecs, StageContext stageContext, Map<String, Map<String, YTreeNode>> labels,
                            List<Pair<String, Boolean>> clusterSequence) {
            this.newSpecs = newSpecs;
            this.stageContext = stageContext;
            this.labels = labels;
            this.clusterSequence = clusterSequence;
        }

        public Pair<String, Boolean> current() {
            if (clusterSequence.isEmpty()) {
                return null;
            }
            return clusterSequence.get(currentIndex);
        }

        public boolean hasNext() {
            return currentIndex + 1 < clusterSequence.size();
        }

        public Pair<String, Boolean> next() {
            if (clusterSequence.isEmpty()) {
                return null;
            }
            if (!hasNext()) {
                return current();
            }
            currentIndex++;
            return clusterSequence.get(currentIndex);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            CurrentStep that = (CurrentStep) o;
            return Objects.equals(newSpecs, that.newSpecs) &&
                    Objects.equals(stageContext.getAcl(), that.stageContext.getAcl()) &&
                    Objects.equals(labels, that.labels) &&
                    Objects.equals(clusterSequence, that.clusterSequence);
        }

        @Override
        public int hashCode() {
            return Objects.hash(newSpecs, stageContext.getAcl(), labels, clusterSequence);
        }
    }

}
