package ru.yandex.infra.stage.deployunit;

import java.util.Optional;
import java.util.function.Consumer;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.stage.dto.DeployProgress;
import ru.yandex.infra.stage.dto.DeploymentStrategy;
import ru.yandex.infra.stage.primitives.DeployPrimitiveStatus;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.RelationController;
import ru.yandex.yp.client.api.EConditionStatus;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.model.YpObjectType;

public class SingleReplicaSetController extends ReplicaSetControllerBase<TReplicaSetSpec, TReplicaSetStatus, String> {
    private final String description;
    private final Converter converter;

    public SingleReplicaSetController(String replicaSetId,
            ObjectLifeCycleManager<SchemaMeta, TReplicaSetSpec,TReplicaSetStatus> repository,
            Consumer<DeployPrimitiveStatus<TReplicaSetStatus>> updateNotifier,
            Converter converter,
            AclUpdater aclUpdater,
            String stageFqid,
            RelationController relationController) {
        super(replicaSetId, repository, updateNotifier, aclUpdater, YpObjectType.REPLICA_SET, stageFqid, relationController);
        this.description = String.format("(%s) replica set %s", repository.getCluster().orElse("xdc"), replicaSetId);
        this.converter = converter;
    }

    @Override
    protected String selfDescription() {
        return description;
    }

    @Override
    protected String extractSpecRevisionId(TReplicaSetSpec spec) {
        return spec.getRevisionId();
    }

    @Override
    protected String extractStatusRevisionId(TReplicaSetStatus status) {
        return status.getRevisionId();
    }

    @Override
    protected DeployProgress extractDeployProgress(TReplicaSetStatus status) {
        return converter.fromProto(status.getDeployStatus().getDetails().getCurrentRevisionProgress());
    }

    @Override
    protected int extractReplicaCount(TReplicaSetSpec spec) {
        return spec.getReplicaCount();
    }

    @Override
    protected boolean isReady(TReplicaSetStatus status) {
        return status.getReadyCondition().getStatus() == EConditionStatus.CS_TRUE;
    }

    @Override
    protected TReplicaSetSpec patchReplicaCount(TReplicaSetSpec spec, int replicaCount) {
        return spec.toBuilder().setReplicaCount(replicaCount).build();
    }

    @Override
    protected DeploymentStrategy extractDeploymentStrategy(TReplicaSetSpec spec) {
        return converter.fromProto(spec.getDeploymentStrategy());
    }

    @Override
    protected Optional<TReplicaSetStatus> emptyYpStatus() {
        return Optional.empty();
    }
}
