package ru.yandex.infra.stage.deployunit;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.protobuf.Message;
import org.slf4j.Logger;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.yp.CreateObjectRequest;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.ObjectLifeCycleEventType;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.SpecStatusMeta;
import ru.yandex.infra.stage.yp.SpecWithAcl;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.yp.model.YpObjectType;

import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;

public abstract class YpObjectControllerBase<DtoMeta extends SchemaMeta, YpSpec extends Message, YpStatus extends Message, Status> implements ObjectController<YpSpec, Status> {

    private final String id;
    protected final ObjectLifeCycleManager<DtoMeta, YpSpec, YpStatus> repository;
    private final Consumer<Status> updateHandler;
    private final AclUpdater aclUpdater;
    private final String typeName;
    private final String typeForLogs;

    protected Optional<SpecWithAcl<YpSpec>> currentSpecOpt = empty();
    protected Map<String, YTreeNode> currentLabels = new HashMap<>();
    protected Map<String, Long> currentTimestampPrerequisites = new HashMap<>();
    protected StageContext currentStageContext;
    private Status currentStatus;

    public YpObjectControllerBase(String id, ObjectLifeCycleManager<DtoMeta, YpSpec, YpStatus> repository,
                                  Consumer<Status> updateHandler,
                                  AclUpdater aclUpdater, YpObjectType type) {
        this.id = id;
        this.repository = repository;
        this.updateHandler = updateHandler;
        this.aclUpdater = aclUpdater;

        currentStatus = getStatusForNewSpec();
        typeName = type.name();
        typeForLogs = type.name().toLowerCase();
    }

    protected abstract Status getStatusForNewSpec();
    protected abstract Status getStatus(Readiness readiness);
    protected abstract void handleCurrentState(Optional<SpecStatusMeta<DtoMeta, YpSpec, YpStatus>> ypObject);
    protected void handleRemove() {}
    protected abstract Logger getLogger();

    protected void handleYpObjectUpdate(Optional<SpecStatusMeta<DtoMeta, YpSpec, YpStatus>> ypObject,
                                        ObjectLifeCycleEventType eventType) {
        if (eventType == ObjectLifeCycleEventType.REMOVED) {
            handleRemove();
        } else {
            handleCurrentState(ypObject);
        }
    }

    @Override
    public void sync(YpSpec ypSpec, StageContext stageContext, Map<String, YTreeNode> labels, Map<String, Long> timestampPrerequisites,
                     String cluster) {
        if (repository.getCluster().isPresent() && cluster != null && !cluster.equals(repository.getCluster().get())) {
            throw new IllegalStateException("Fail to apply spec. Invalid cluster name or controller");
        }
        if (currentSpecOpt.isEmpty()) {
            getLogger().info("Starting controller for {} {}", typeForLogs, id);
            repository.startManaging(id, this::handleYpObjectUpdate);
        }
        Acl acl = stageContext.getAcl();
        currentStageContext = stageContext;
        currentSpecOpt = Optional.of(new SpecWithAcl<>(ypSpec, aclUpdater.update(acl)));
        currentLabels = labels;
        currentStatus = getStatusForNewSpec();
        currentTimestampPrerequisites = timestampPrerequisites;
    }

    @Override
    public Status getStatus() {
        return currentStatus;
    }

    @Override
    public void shutdown() {
        getLogger().info("Stopping controller for {} {}", typeForLogs, id);
        repository.stopManaging(id);
    }

    protected String getReplicaSetId() {
        return id;
    }

    protected void updateObjectSpec(YpSpec spec) {
        updateObjectSpecWithLabels(spec, emptyMap(), empty());
    }

    protected void updateObjectSpecWithLabels(YpSpec spec, Map<String, YTreeNode> labels,
                                              Optional<Long> prerequisiteTimestamp) {
        repository.updateSpecWithLabels(id, spec, labels, prerequisiteTimestamp,
                () -> updateStatus(getStatus(Readiness.inProgress(typeName + "_UPDATED"))),
                error -> {
                    if (!checkHash(Source.SPEC, error.getMessage())) {
                        getLogger().error(String.format("Could not update %s %s", typeForLogs, id), error);
                        updateStatus(getStatus(Readiness.failed(typeName + "_UPDATE_FAILED", ExceptionUtils
                                .getAllMessages(error))));
                    }
                }
        );
    }

    protected void updateObjectAcl(Acl acl, Optional<Long> metaPrerequisiteTimestamp) {
        repository.updateAcl(id, acl, metaPrerequisiteTimestamp,
                () -> getLogger().info("Acl for {} {} updated successfully", typeForLogs, id),
                error -> {
                    if (!checkHash(Source.ACL, error.getMessage())) {
                        getLogger().error(String.format("Could not update acl for %s %s", typeForLogs, id), error);
                    }
                }
        );
    }

    protected void updateObjectAcl(Acl acl) {
        updateObjectAcl(acl, empty());
    }

    protected void removeLabel(String label) {
        repository.removeLabel(id, label,
                () -> getLogger().info("Label {} for {} {} was removed successfully", label, typeForLogs, id),
                error -> {
                    if (!checkHash(Source.REMOVE_LABEL, error.getMessage())) {
                        getLogger().error(String.format("Could not remove label %s for %s %s", label, typeForLogs, id), error);
                    }
                }
        );
    }

    protected void createObject(Function<YpObjectTransactionalRepository<DtoMeta,YpSpec,YpStatus>, CompletableFuture<?>> createAction) {
        createObject((onSuccess, onFailure) -> repository.create(id, createAction, onSuccess, onFailure));
    }

    protected void createObject(CreateObjectRequest<YpSpec> request) {
        createObject((onSuccess, onFailure) -> repository.create(id, request, onSuccess, onFailure));
    }

    private void createObject(BiConsumer<Runnable,Consumer<Throwable>> createAction) {
        getLogger().info("Creating {} {}", typeForLogs, id);
        createAction.accept(
                () -> updateStatus(getStatus(Readiness.inProgress(typeName + "_CREATED"))),
                error -> {
                    if (!checkHash(Source.CREATE, error.getMessage())) {
                        getLogger().warn("Could not create {} {}", typeForLogs, id, error);
                        updateStatus(getStatus(Readiness.failed(typeName + "_CREATION_FAILED", ExceptionUtils.getAllMessages(error))));
                    }
                });
    }

    protected CreateObjectRequest.Builder<YpSpec> constructSimpleCreateRequestBuilder(SpecWithAcl<YpSpec> specWithAcl) {
        return new CreateObjectRequest.Builder<>(specWithAcl.getSpec()).setAcl(specWithAcl.getAcl());
    }

    protected void updateStatus(Status newStatus) {
        if (!currentStatus.equals(newStatus)) {
            currentStatus = newStatus;
            updateHandler.accept(newStatus);
        }
    }

    private boolean checkHash(Source source, String description){
        final int hash = Objects.hash(id, currentLabels, currentSpecOpt, source);
        final boolean inCache = repository.isInCache(hash);
        if (inCache) {
            repository.addError(hash, description);
        }
        return inCache;

    }
    private enum Source {
        CREATE, SPEC, ACL, REMOVE_LABEL
    }
}
