package ru.yandex.infra.stage.docker;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.util.JsonHttpGetter;

public class DockerHttpGetterImpl implements DockerGetter {
    private final static String REQUEST_TEMPLATE = "/api/docker/resolve?registryUrl=%s/%s&tag=%s";
    private final static Logger LOG = LoggerFactory.getLogger(DockerHttpGetterImpl.class);

    private final String templateUrl;
    private final String defaultRegistryHost;
    private final JsonHttpGetter jsonHttpGetter;

    public DockerHttpGetterImpl(String host, String defaultRegistryHost, JsonHttpGetter jsonHttpGetter) {
        this.defaultRegistryHost = defaultRegistryHost;
        this.templateUrl = host + REQUEST_TEMPLATE;
        this.jsonHttpGetter = jsonHttpGetter;
    }

    @Override
    public CompletableFuture<DockerImageContents> get(DockerImageDescription description) {
        String registryHost = Optional.ofNullable(description.getRegistryHost()).orElse(defaultRegistryHost);
        String url = String.format(templateUrl, registryHost, description.getName(), description.getTag());

        return jsonHttpGetter.get(url)
                .thenApply(node -> parseResponse(node, description));
    }

    private DockerImageContents parseResponse(JsonNode node, DockerImageDescription description) {
        try {
            return DockerImageContents.fromJson(node, description);
        } catch (Exception e) {
            throw new DockerInfoJSONParseException(String.format("Invalid docker_info JSON. %s", e.getMessage()));
        }
    }
}
