package ru.yandex.infra.stage.docker;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.util.ExceptionUtils;
import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.util.AdaptiveRateLimiter;
import ru.yandex.infra.stage.util.GeneralUtils;
import ru.yandex.infra.stage.util.StoppableBackOff;
import ru.yandex.infra.stage.util.StoppableBackOffExecution;

// Must be used from serializer thread only
public class DockerImagesResolverImpl implements DockerImagesResolver {

    private static final Logger LOG = LoggerFactory.getLogger(DockerImagesResolverImpl.class);

    private final static String DOCKER_RESOLVING_REASON = "DOCKER_RESOLVING";
    private final static String PARSE_FAIL_REASON = "DOCKER_INFO_PARSING_ERROR";
    private final static String OTHER_FAIL_REASON = "DOCKER_RESOLVER_ERROR";

    private static final Pattern DOCKER_NAME_PATTERN = Pattern.compile("[a-z0-9-_.:/]+");
    private static final Pattern DOCKER_TAG_PATTERN = Pattern.compile("[A-Za-z0-9-_.:]+");

    static final String METRIC_SCHEDULED_REQUESTS_COUNT = "scheduled_requests_count";
    static final String METRIC_SEND_REQUESTS_COUNT = "send_requests_count";
    static final String METRIC_FORCE_RESOLVE_COUNT = "force_resolve_count";
    static final String METRIC_FAILED_REQUESTS_COUNT = "failed_requests_count";
    static final String METRIC_SUCCEED_REQUESTS_COUNT = "succeed_requests_count";
    static final String METRIC_IMAGES_COUNT = "images_count";
    static final String METRIC_ACTIVE_REQUESTS_COUNT = "active_requests_count";

    private final SerialExecutor serialExecutor;
    private final DockerGetter dockerGetter;
    //ConcurrentHashMap, because it is accessed also by metrics lib from another thread
    private final Map<DockerImageDescription, DockerResolveStatus> resolveStatuses = new ConcurrentHashMap<>();
    private final Map<String, DockerResolveResultHandler> resultHandlers = new HashMap<>();
    private final Multimap<DockerImageDescription, String> subscribers = HashMultimap.create();
    private final StoppableBackOff timeoutFactory;
    private final Cache<DockerImageContents> persistence;
    private final boolean allowForceResolve;
    private final AdaptiveRateLimiter rateLimiter;

    private final AtomicLong metricScheduledRequestsCount = new AtomicLong();
    private final AtomicLong metricSendRequestsCount = new AtomicLong();
    private final AtomicLong metricForceResolveCount = new AtomicLong();
    private final AtomicLong metricFailedRequestsCount = new AtomicLong();
    private final AtomicLong metricSucceedRequestsCount = new AtomicLong();

    public DockerImagesResolverImpl(SerialExecutor serialExecutor,
                                    DockerGetter dockerGetter,
                                    Duration initialRetryInterval,
                                    Duration maxRetryInterval,
                                    Cache<DockerImageContents> persistence,
                                    boolean allowForceResolve,
                                    GaugeRegistry registry,
                                    AdaptiveRateLimiter rateLimiter) {
        this.dockerGetter = dockerGetter;
        this.serialExecutor = serialExecutor;
        this.timeoutFactory = GeneralUtils.getTimeoutFactory(initialRetryInterval.toMillis(), maxRetryInterval.toMillis());
        this.persistence = persistence;
        this.allowForceResolve = allowForceResolve;
        this.rateLimiter = rateLimiter;
        // It is safer to ensure state presence here and explicitly set an empty state in provider if necessary
        persistence.getAll().values().forEach(contents -> resolveStatuses.put(contents.getDescription(), DockerResolveStatus.getReadyStatus(contents)));

        registry.add(METRIC_IMAGES_COUNT, resolveStatuses::size);
        registry.add(METRIC_SCHEDULED_REQUESTS_COUNT, new GolovanableGauge<>(metricScheduledRequestsCount::get, "ammx"));
        registry.add(METRIC_SEND_REQUESTS_COUNT, new GolovanableGauge<>(metricSendRequestsCount::get, "dmmm"));
        registry.add(METRIC_FORCE_RESOLVE_COUNT, new GolovanableGauge<>(metricForceResolveCount::get, "dmmm"));
        registry.add(METRIC_FAILED_REQUESTS_COUNT, new GolovanableGauge<>(metricFailedRequestsCount::get, "dmmm"));
        registry.add(METRIC_SUCCEED_REQUESTS_COUNT, new GolovanableGauge<>(metricSucceedRequestsCount::get, "dmmm"));
        registry.add(METRIC_ACTIVE_REQUESTS_COUNT, new GolovanableGauge<>(rateLimiter::getActiveRequestsCount, "ammx"));
    }

    @Override
    public void registerResultHandler(String unitId, DockerResolveResultHandler handler) {
        resultHandlers.put(unitId, handler);
    }

    @Override
    public void unregisterResultHandler(String unitId) {
        resultHandlers.remove(unitId);
    }

    @Override
    public void addResolve(DockerImageDescription description, String unitId) {
        subscribers.put(description, unitId);
        if (!resolveStatuses.containsKey(description)) {
            doResolve(description);
        }
    }

    @Override
    public void forceResolve(DockerImageDescription description) {
        if (!allowForceResolve) {
            LOG.debug("forceResolve is disabled by config entry (docker_resolver.allow_force_resolve = false)");
            return;
        }
        metricForceResolveCount.incrementAndGet();
        doResolve(description);
    }

    private void doResolve(DockerImageDescription description) {
        String message = String.format("Resolving docker image %s", getImageKeyForCache(description));
        resolveStatuses.put(description, DockerResolveStatus.getInProgressStatus(DOCKER_RESOLVING_REASON, message));
        metricScheduledRequestsCount.incrementAndGet();
        resolveOrRetry(description);
    }

    private void resolveOrRetry(DockerImageDescription description) {
        if (!rateLimiter.tryAcquire()) {
            serialExecutor.schedule(() -> {
                if (subscribers.containsKey(description)) {
                    resolveOrRetry(description);
                } else {
                    metricScheduledRequestsCount.decrementAndGet();
                    resolveStatuses.remove(description);
                }
            }, Duration.ofSeconds(1));
            return;
        }
        final StoppableBackOffExecution backOffExecution = timeoutFactory.startStoppable();
        serialExecutor.executeOrRetry(() -> {
                LOG.info("Resolving docker image {} for {}", description, subscribersString(description));
                metricSendRequestsCount.incrementAndGet();
                rateLimiter.incrementAndGet();
                return dockerGetter.get(description)
                        .whenComplete((x, error) -> rateLimiter.decrementAndGet());
            }, result -> {
                metricScheduledRequestsCount.decrementAndGet();
                metricSucceedRequestsCount.incrementAndGet();
                addResult(description, result);
            }, throwable -> {
                metricFailedRequestsCount.incrementAndGet();
                Throwable cause = ExceptionUtils.stripCompletionException(throwable);
                if (subscribers.containsKey(description)) {
                    String reason = OTHER_FAIL_REASON;
                    if (cause instanceof DockerInfoJSONParseException) {
                        reason = PARSE_FAIL_REASON;
                    }
                    resolveStatuses.put(description, DockerResolveStatus.getFailedStatus(reason,
                            String.format("Could not resolve '%s': %s",
                                    getImageKeyForCache(description),
                                    cause.getMessage())));
                    Collection<String> currentSubscribers = subscribers.get(description);
                    currentSubscribers.retainAll(resultHandlers.keySet());
                    copyAndForEach(currentSubscribers,
                            unitId -> resultHandlers.get(unitId).onDockerResolveFailure(description, cause));
                } else {
                    metricScheduledRequestsCount.decrementAndGet();
                    resolveStatuses.remove(description);
                    backOffExecution.stop();
                }
            }, backOffExecution);
    }

    @Override
    public void removeResolve(DockerImageDescription description, String unitId) {
        // we expect that this method is called only after state restoration has completed,
        // so all subscriber refs are in place
        subscribers.remove(description, unitId);
        if (!subscribers.containsKey(description)) {
            LOG.info("Removing docker image {} from persistence", description);
            resolveStatuses.remove(description);
            persistence.remove(getImageKeyForCache(description));
        }
    }

    @Override
    public DockerResolveStatus getResolveStatus(DockerImageDescription description) {
        DockerResolveStatus result = resolveStatuses.get(description);
        if (result == null) {
            return DockerResolveStatus.getEmptyStatus();
        }
        return result;
    }

    private void addResult(DockerImageDescription description, DockerImageContents contents) {
        DockerResolveStatus status = resolveStatuses.get(description);
        if (status == null || status.getResult().isEmpty()) {
            resolveStatuses.put(description, DockerResolveStatus.getReadyStatus(contents));
            Collection<String> currentSubscribers = subscribers.get(description);
            currentSubscribers.retainAll(resultHandlers.keySet());
            copyAndForEach(currentSubscribers,
                    unitId -> resultHandlers.get(unitId).onDockerResolveSuccess(description, contents));
            LOG.info("Adding docker image {} to persistence", description);
            persistence.put(getImageKeyForCache(description), contents);
        }
    }

    private void copyAndForEach(Collection<String> currentSubscribers, Consumer<String> function) {
        new ArrayList<>(currentSubscribers).forEach(function);
    }

    @VisibleForTesting
    public static String getImageKeyForCache(DockerImageDescription description) {
        return String.format("%s/%s:%s", description.getRegistryHost(), description.getName(), description.getTag());
    }

    public static List<String> validateDescription(DockerImageDescription description) {
        List<String> result = new LinkedList<>();

        String name = description.getName();
        if (!matchesPattern(name, DOCKER_NAME_PATTERN)) {
            result.add(String.format("Docker image name '%s' must match pattern '%s'", name, DOCKER_NAME_PATTERN));
        }

        String tag = description.getTag();
        if (!matchesPattern(tag, DOCKER_TAG_PATTERN)) {
            result.add(String.format("Docker image tag '%s' must match pattern '%s'", tag, DOCKER_TAG_PATTERN));
        }

        return result;
    }

    private String subscribersString(DockerImageDescription description) {
        return Arrays.toString(subscribers.get(description).toArray());
    }

    private static boolean matchesPattern(String string, Pattern pattern) {
        return pattern.matcher(string).matches();
    }

}
