package ru.yandex.infra.stage.dto;

import java.util.Objects;

import com.google.common.base.MoreObjects;

import ru.yandex.yp.client.pods.TComputeResources;
import ru.yandex.yp.client.pods.TComputeResourcesOrBuilder;

public class AllComputeResources {
    public static final long UNKNOWN_DISK_CAPACITY = -1;

    // 1000 vcpu = 1 real core
    // 1 memory (or disk capacity) unit = 1 byte
    private final long vcpuGuarantee;
    private final long vcpuLimit;
    private final long memoryGuarantee;
    private final long memoryLimit;
    private final long anonymousMemoryLimit;
    private final long diskCapacity;
    private final long threadLimit;

    public AllComputeResources(long vcpuGuarantee,
                               long vcpuLimit,
                               long memoryGuarantee,
                               long memoryLimit,
                               long anonymousMemoryLimit,
                               long diskCapacity,
                               long threadLimit) {
        this.vcpuGuarantee = vcpuGuarantee;
        this.vcpuLimit = vcpuLimit;
        this.memoryGuarantee = memoryGuarantee;
        this.memoryLimit = memoryLimit;
        this.anonymousMemoryLimit = anonymousMemoryLimit;
        this.diskCapacity = diskCapacity;
        this.threadLimit = threadLimit;
    }

    public AllComputeResources(long vcpu,
                               long memory,
                               long diskCapacity,
                               long threadLimit) {
        this(
            vcpu,
            memory,
            memory,
            diskCapacity,
            threadLimit
        );
    }

    public AllComputeResources(long vcpu,
                               long memory,
                               long anonymousMemoryLimit,
                               long diskCapacity,
                               long threadLimit) {
        this(
            vcpu,
            vcpu,
            memory,
            memory,
            anonymousMemoryLimit,
            diskCapacity,
            threadLimit
        );
    }

    public AllComputeResources(TComputeResourcesOrBuilder resources,
                               long diskCapacity,
                               long threadLimit) {
        this(
            resources.getVcpuGuarantee(),
            resources.getVcpuLimit(),
            resources.getMemoryGuarantee(),
            resources.getMemoryLimit(),
            resources.getAnonymousMemoryLimit(),
            diskCapacity,
            threadLimit
        );
    }

    public long getVcpuGuarantee() {
        return vcpuGuarantee;
    }

    public long getVcpuLimit() {
        return vcpuLimit;
    }

    public long getMemoryGuarantee() {
        return memoryGuarantee;
    }

    public long getMemoryLimit() {
        return memoryLimit;
    }

    public long getAnonymousMemoryLimit() {
        return anonymousMemoryLimit;
    }

    public long getDiskCapacity() {
        return diskCapacity;
    }

    public long getThreadLimit() { return threadLimit; }

    public AllComputeResources multiply(int multiplier) {
        return new AllComputeResources(
                vcpuGuarantee * multiplier,
                vcpuLimit * multiplier,
                memoryGuarantee * multiplier,
                memoryLimit * multiplier,
                anonymousMemoryLimit * multiplier,
                diskCapacity * multiplier,
                threadLimit * multiplier
        );
    }

    public TComputeResources toProto() {
        TComputeResources.Builder builder = TComputeResources.newBuilder()
                .setVcpuGuarantee(vcpuGuarantee)
                .setVcpuLimit(vcpuLimit)
                .setMemoryGuarantee(memoryGuarantee)
                .setMemoryLimit(memoryLimit)
                .setAnonymousMemoryLimit(anonymousMemoryLimit);

        if (threadLimit != 0) {
            builder.setThreadLimit(threadLimit);
        }

        return builder.build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof AllComputeResources)) return false;
        AllComputeResources that = (AllComputeResources) o;
        return vcpuGuarantee == that.vcpuGuarantee &&
                vcpuLimit == that.vcpuLimit &&
                memoryGuarantee == that.memoryGuarantee &&
                memoryLimit == that.memoryLimit &&
                anonymousMemoryLimit == that.anonymousMemoryLimit &&
                diskCapacity == that.diskCapacity &&
                threadLimit == that.threadLimit;
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                vcpuGuarantee,
                vcpuLimit,
                memoryGuarantee,
                memoryLimit,
                anonymousMemoryLimit,
                diskCapacity,
                threadLimit
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("vcpuGuarantee", vcpuGuarantee)
                .add("vcpuLimit", vcpuLimit)
                .add("memoryGuarantee", memoryGuarantee)
                .add("memoryLimit", memoryLimit)
                .add("anonymousMemoryLimit", anonymousMemoryLimit)
                .add("diskCapacity", diskCapacity)
                .add("threadLimit", threadLimit)
                .toString();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private long vcpuGuarantee;
        private long vcpuLimit;
        private long memoryGuarantee;
        private long memoryLimit;
        private long anonymousMemoryLimit;
        private long diskCapacity;
        private long threadLimit;

        Builder(AllComputeResources resources) {
            this.vcpuGuarantee = resources.vcpuGuarantee;
            this.vcpuLimit = resources.vcpuLimit;
            this.memoryGuarantee = resources.memoryGuarantee;
            this.memoryLimit = resources.memoryLimit;
            this.anonymousMemoryLimit = resources.anonymousMemoryLimit;
            this.diskCapacity = resources.diskCapacity;
            this.threadLimit = resources.threadLimit;
        }

        public Builder withVcpuGuarantee(long vcpuGuarantee) {
            this.vcpuGuarantee = vcpuGuarantee;
            return this;
        }

        public Builder withVcpuLimit(long vcpuLimit) {
            this.vcpuLimit = vcpuLimit;
            return this;
        }

        public Builder withMemoryGuarantee(long memoryGuarantee) {
            this.memoryGuarantee = memoryGuarantee;
            return this;
        }

        public Builder withMemoryLimit(long memoryLimit) {
            this.memoryLimit = memoryLimit;
            return this;
        }

        public Builder withAnonymousMemoryLimit(long anonymousMemoryLimit) {
            this.anonymousMemoryLimit = anonymousMemoryLimit;
            return this;
        }

        Builder withDiskCapacity(long diskCapacity) {
            this.diskCapacity = diskCapacity;
            return this;
        }

        Builder withThreadLimit(long threadLimit) {
            this.threadLimit = threadLimit;
            return this;
        }

        public AllComputeResources build() {
            return new AllComputeResources(
                    vcpuGuarantee,
                    vcpuLimit,
                    memoryGuarantee,
                    memoryLimit,
                    anonymousMemoryLimit,
                    diskCapacity,
                    threadLimit
            );
        }
    }
}
