package ru.yandex.infra.stage.dto;

import java.util.Objects;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import com.google.common.base.MoreObjects;

public class Checksum {
    private static final String SCHEME_SEPARATOR = ":";
    private static final Predicate<String> CHECKER = Pattern.compile("^[a-z0-9]*$").asMatchPredicate();
    public static final Checksum EMPTY = new Checksum("", Type.EMPTY);

    public enum Type {
        EMPTY("EMPTY"),
        MD5("MD5"),
        SHA256("SHA256"),
        UNRECOGNIZED("UNRECOGNIZED");

        private final String schemeName;

        Type(String schemeName) {
            this.schemeName = schemeName;
        }

        public String schemeName() {
            return schemeName;
        }
    }

    private Type type;
    private String value;

    public Checksum(String value, Type type) {
        if (value.contains(SCHEME_SEPARATOR)) {
            throw new IllegalArgumentException("Scheme prefix is not allowed by flag");
        }
        this.type = type;
        this.value = value;
    }

    public static Checksum fromString(String value) {
        String[] parts = value.split(SCHEME_SEPARATOR, 2);
        String typePart = parts[0].toUpperCase();
        Type type = calcType(typePart);
        if (parts.length < 2) {
            throw new IllegalArgumentException(String.format("Incorrect '%s' in checksum %s", SCHEME_SEPARATOR, value));
        }
        String valuePart = parts[1];
        if (!CHECKER.test(valuePart)) {
            throw new IllegalArgumentException(String.format("Invalid value for checksum %s", value));
        }
        return new Checksum(valuePart, type);
    }

    private static Type calcType(String type) {
        if (type.equals("SHA256")) {
            return Type.SHA256;
        } else if (type.equals("MD5")) {
            return Type.MD5;
        } if (type.equals("EMPTY")) {
            return Type.EMPTY;
        }
        throw new IllegalArgumentException("Checksum must start with sha256, md5 or empty");
    }

    public String getValue() {
        return value;
    }

    public Type getType() { return type; }

    public String toAgentFormat() {
        return String.format("%s%s%s", type.schemeName(), SCHEME_SEPARATOR, value);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("type", type)
                .add("value", value)
                .toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Checksum checksum = (Checksum) o;
        return value.equals(checksum.value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(value);
    }
}
