package ru.yandex.infra.stage.dto;

import java.util.Objects;
import java.util.Optional;

public class CoredumpConfig {
    private final int countLimit;
    private final int totalSizeLimit;
    private final int probability;
    private final int ttlSeconds;
    private final boolean aggregatorIsEnabled;
    private final String aggregatorUrl;
    private final Optional<String> serviceName;
    private final Optional<String> ctype;
    private final Optional<CoredumpOutputPolicy> outputPolicy;

    public CoredumpConfig(int countLimit,
                          int totalSizeLimit,
                          int probability,
                          int ttlSeconds,
                          boolean aggregatorIsEnabled,
                          String aggregatorUrl,
                          Optional<String> serviceName,
                          Optional<String> ctype,
                          Optional<CoredumpOutputPolicy> outputPolicy) {
        this.countLimit = countLimit;
        this.totalSizeLimit = totalSizeLimit;
        this.probability = probability == 0 ? 100 : probability; // 100% by default
        this.ttlSeconds = ttlSeconds;
        this.aggregatorIsEnabled = aggregatorIsEnabled;
        this.aggregatorUrl = aggregatorUrl == null ? "" : aggregatorUrl;
        this.serviceName = serviceName;
        this.ctype = ctype;
        this.outputPolicy = outputPolicy;
    }

    public int getCountLimit() {
        return countLimit;
    }

    public int getTotalSizeLimit() {
        return totalSizeLimit;
    }

    public int getProbability() {
        return probability;
    }

    public int getTtlSeconds() {
        return ttlSeconds;
    }

    public boolean isAggregatorEnabled() {
        return aggregatorIsEnabled;
    }

    public String getAggregatorUrl() {
        return aggregatorUrl;
    }

    public Optional<String> getServiceName() {
        return serviceName;
    }

    public Optional<String> getCtype() {
        return ctype;
    }

    public Optional<CoredumpOutputPolicy> getOutputPolicy() {
        return outputPolicy;
    }

    public CoredumpConfig withOutputPolicy(CoredumpOutputPolicy outputPolicy) {
        return toBuilder().withOutputPolicy(outputPolicy).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CoredumpConfig that = (CoredumpConfig) o;
        return countLimit == that.countLimit &&
                totalSizeLimit == that.totalSizeLimit &&
                probability == that.probability &&
                ttlSeconds == that.ttlSeconds &&
                aggregatorIsEnabled == that.aggregatorIsEnabled &&
                aggregatorUrl.equals(that.aggregatorUrl) &&
                serviceName.equals(that.serviceName) &&
                ctype.equals(that.ctype) &&
                outputPolicy.equals(that.outputPolicy);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                countLimit,
                totalSizeLimit,
                probability,
                ttlSeconds,
                aggregatorIsEnabled,
                aggregatorUrl,
                serviceName,
                ctype,
                outputPolicy
        );
    }

    private Builder toBuilder() {
        return new Builder(this);
    }

    private static class Builder {
        private final int countLimit;
        private final int totalSizeLimit;
        private final int probability;
        private final int ttlSeconds;
        private final boolean aggregatorIsEnabled;
        private final String aggregatorUrl;
        private final Optional<String> serviceName;
        private final Optional<String> ctype;
        private Optional<CoredumpOutputPolicy> outputPolicy;

        Builder(CoredumpConfig config) {
            this.countLimit = config.countLimit;
            this.totalSizeLimit = config.totalSizeLimit;
            this.probability = config.probability;
            this.ttlSeconds = config.ttlSeconds;
            this.aggregatorIsEnabled = config.aggregatorIsEnabled;
            this.aggregatorUrl = config.aggregatorUrl;
            this.serviceName = config.serviceName;
            this.ctype = config.ctype;
            this.outputPolicy = config.outputPolicy;
        }

        CoredumpConfig build() {
            return new CoredumpConfig(
                    countLimit,
                    totalSizeLimit,
                    probability,
                    ttlSeconds,
                    aggregatorIsEnabled,
                    aggregatorUrl,
                    serviceName,
                    ctype,
                    outputPolicy
            );
        }

        Builder withOutputPolicy(Optional<CoredumpOutputPolicy> outputPolicy) {
            this.outputPolicy = outputPolicy;
            return this;
        }

        Builder withOutputPolicy(CoredumpOutputPolicy outputPolicy) {
            return withOutputPolicy(Optional.ofNullable(outputPolicy));
        }
    }
}
