package ru.yandex.infra.stage.dto;

import java.util.Objects;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;

import ru.yandex.infra.stage.deployunit.DeployUnitTimeline;

public class DeployUnitStatus {
    private final Condition inProgress;
    private final Condition failure;
    private final DeployUnitSpec currentTarget;
    private final DeployUnitStatusDetails details;
    private final DeployProgress deployProgress;
    private final String yasmItype;
    private final DeployUnitTimeline deployUnitTimeline;

    public DeployUnitStatus(Condition inProgress,
                            Condition failure,
                            DeployUnitSpec currentTarget,
                            DeployUnitStatusDetails details,
                            DeployProgress deployProgress,
                            String yasmItype,
                            DeployUnitTimeline deployUnitTimeline) {
        this.inProgress = inProgress;
        this.failure = failure;
        this.currentTarget = currentTarget;
        this.details = details;
        this.deployProgress = deployProgress;
        this.yasmItype = yasmItype;
        this.deployUnitTimeline = deployUnitTimeline;
    }

    public Condition getReady() {
        return deployUnitTimeline.getLatestReadyCondition();
    }

    public Condition getInProgress() {
        return inProgress;
    }

    public Condition getFailure() {
        return failure;
    }

    public DeployUnitSpec getCurrentTarget() {
        return currentTarget;
    }

    public long getTargetRevision() {
        return deployUnitTimeline.getTargetRevision();
    }

    public DeployUnitStatusDetails getDetails() {
        return details;
    }

    public DeployProgress getDeployProgress() {
        return deployProgress;
    }

    public long getTargetSpecTimestamp() {
        return deployUnitTimeline.getStartTimestamp().toEpochMilli();
    }

    public String getYasmItype() {
        return yasmItype;
    }

    public DeployUnitTimeline getDeployUnitTimeline() {
        return deployUnitTimeline;
    }

    public long getLatestDeployedRevision() {
        return deployUnitTimeline.getLatestDeployedRevision();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DeployUnitStatus)) {
            return false;
        }
        DeployUnitStatus that = (DeployUnitStatus) o;
        return Objects.equals(inProgress, that.inProgress) &&
                Objects.equals(failure, that.failure) &&
                Objects.equals(currentTarget, that.currentTarget) &&
                Objects.equals(details, that.details) &&
                Objects.equals(deployProgress, that.deployProgress) &&
                Objects.equals(yasmItype, that.yasmItype) &&
                Objects.equals(deployUnitTimeline, that.deployUnitTimeline);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                inProgress,
                failure,
                currentTarget,
                details,
                deployProgress,
                yasmItype,
                deployUnitTimeline
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("ready", getReady())
                .add("inProgress", getInProgress())
                .add("failure", getFailure())
                .add("currentTarget", getCurrentTarget())
                .add("targetRevision", getTargetRevision())
                .add("details", getDetails())
                .add("deployProgress", getDeployProgress())
                .add("targetSpecTimestamp", getTargetSpecTimestamp())
                .add("yasmItype", getYasmItype())
                .add("deployUnitTimeline", getDeployUnitTimeline())
                .add("latestDeployedRevision", getLatestDeployedRevision())
                .toString();
    }

    public DeployUnitStatus withDeployUnitTimeline(DeployUnitTimeline deployUnitTimeline) {
        return toBuilder().withDeployUnitTimeline(deployUnitTimeline).build();
    }

    @VisibleForTesting
    public DeployUnitStatus withDetails(DeployUnitStatusDetails details) {
        return toBuilder().withDetails(details).build();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private final Condition inProgress;
        private final Condition failure;
        private DeployUnitSpec currentTarget;
        private DeployUnitStatusDetails details;
        private final DeployProgress deployProgress;
        private final String yasmItype;
        private DeployUnitTimeline deployUnitTimeline;

        Builder(DeployUnitStatus status) {
            this.inProgress = status.inProgress;
            this.failure = status.failure;
            this.currentTarget = status.currentTarget;
            this.details = status.details;
            this.deployProgress = status.deployProgress;
            this.yasmItype = status.yasmItype;
            this.deployUnitTimeline = status.deployUnitTimeline;
        }

        public DeployUnitStatus build() {
            return new DeployUnitStatus(
                    inProgress,
                    failure,
                    currentTarget,
                    details,
                    deployProgress,
                    yasmItype,
                    deployUnitTimeline
            );
        }

        public Builder withCurrentTarget(DeployUnitSpec currentTarget) {
            this.currentTarget = currentTarget;
            return this;
        }

        public Builder withDeployUnitTimeline(DeployUnitTimeline deployUnitTimeline) {
            this.deployUnitTimeline = deployUnitTimeline;
            return this;
        }

        Builder withDetails(DeployUnitStatusDetails details) {
            this.details = details;
            return this;
        }
    }
}
