package ru.yandex.infra.stage.dto;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.MoreObjects;
import one.util.streamex.StreamEx;

import ru.yandex.infra.stage.util.OptionalUtils;

public class DockerImageContents {
    private final DockerImageDescription description;
    private final List<DownloadableResource> layers;
    private final List<String> command;
    private final List<String> entryPoint;
    private final Optional<String> user;
    private final Optional<String> group;
    private final Optional<String> workingDir;
    private final SortedMap<String, String> environment;
    private final Optional<String> digest;

    public DockerImageContents(DockerImageDescription description,
                               List<DownloadableResource> layers,
                               List<String> command,
                               List<String> entryPoint,
                               Optional<String> user,
                               Optional<String> group,
                               Optional<String> workingDir,
                               Map<String, String> environment,
                               Optional<String> digest) {
        this.description = description;
        this.layers = layers;
        this.command = command;
        this.entryPoint = entryPoint;
        this.user = user;
        this.group = group;
        this.workingDir = workingDir;
        this.environment = new TreeMap<>(environment);
        this.digest = digest;
    }

    public DockerImageDescription getDescription() {
        return description;
    }

    public List<DownloadableResource> getLayers() {
        return layers;
    }

    public List<String> getCommand() {
        return command;
    }

    public List<String> getEntryPoint() {
        return entryPoint;
    }

    public Optional<String> getUser() {
        return user;
    }

    public Optional<String> getGroup() {
        return group;
    }

    public Optional<String> getWorkingDir() {
        return workingDir;
    }

    public SortedMap<String, String> getEnvironment() {
        return environment;
    }

    public Optional<String> getDigest() {
        return digest;
    }

    public static DockerImageContents fromJson(JsonNode node, DockerImageDescription descriptionFromRequest) {
        JsonNode layers = node.get("layers2");
        List<DownloadableResource> layersResult = new ArrayList<>();
        for (int i = 0; i < layers.size(); ++i) {
            JsonNode current = layers.get(i);
            String value = current.get("digest").textValue();
            JsonNode torrentUrl = current.get("torrent");
            JsonNode httpUrl = current.get("http");
            JsonNode url = (torrentUrl != null && torrentUrl.textValue() != null) ? torrentUrl : httpUrl;
            layersResult.add(new DownloadableResource(url.textValue(), Checksum.fromString(value)));
        }
        JsonNode dockerConfig = node.get("dockerConfig");
        List<String> command = new ArrayList<>();
        dockerConfig.get("command").forEach(commandNode -> command.add(commandNode.textValue()));
        List<String> entryPoint = new ArrayList<>();
        dockerConfig.get("entryPoint").forEach(entryPointNode -> entryPoint.add(entryPointNode.textValue()));
        String userAndGroup = dockerConfig.get("user").textValue();
        Optional<String> user = Optional.empty();
        Optional<String> group = Optional.empty();
        if (userAndGroup != null) {
            String[] userAndGroupArray = userAndGroup.split(":", 2);
            user = OptionalUtils.emptyStringAsEmptyOptional(userAndGroupArray[0]);
            if (userAndGroupArray.length == 2) {
                group = Optional.of(userAndGroupArray[1]);
            }
        }

        Optional<String> workingDir = OptionalUtils.emptyStringAsEmptyOptional(dockerConfig.get("workingDir").textValue());

        JsonNode environmentNode = dockerConfig.get("environment");
        SortedMap<String, String> environment = StreamEx.of(environmentNode.fieldNames())
                                                        .toSortedMap(k -> k, key -> environmentNode.get(key).textValue());

        Optional<String> digest = Optional.of(node.get("hash").textValue());

        return new DockerImageContents(descriptionFromRequest,
                layersResult, command, entryPoint, user, group, workingDir, environment, digest);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("description", description)
                .add("layers", layers)
                .add("command", command)
                .add("entryPoint", entryPoint)
                .add("user", user)
                .add("group", group)
                .add("workingDir", workingDir)
                .add("environment", environment)
                .add("digest", digest)
                .toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DockerImageContents)) {
            return false;
        }
        DockerImageContents that = (DockerImageContents) o;
        return Objects.equals(description, that.description) &&
                Objects.equals(layers, that.layers) &&
                Objects.equals(command, that.command) &&
                Objects.equals(entryPoint, that.entryPoint) &&
                Objects.equals(user, that.user) &&
                Objects.equals(group, that.group) &&
                Objects.equals(workingDir, that.workingDir) &&
                Objects.equals(environment, that.environment) &&
                Objects.equals(digest, that.digest);
    }

    @Override
    public int hashCode() {
        return Objects.hash(description, layers, command, entryPoint, user, group, workingDir, environment, digest);
    }
}
