package ru.yandex.infra.stage.dto;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;

import ru.yandex.infra.stage.deployunit.ReadyStatus;

public class DynamicResourceRevisionStatus implements ReadyStatus {
    private final long revision;
    private final AggregatedCondition ready;
    private final AggregatedCondition inProgress;
    private final AggregatedCondition error;

    public DynamicResourceRevisionStatus(long revision,
                                         AggregatedCondition ready,
                                         AggregatedCondition inProgress,
                                         AggregatedCondition error) {
        this.revision = revision;
        this.ready = ready;
        this.inProgress = inProgress;
        this.error = error;
    }

    public long getRevision() {
        return revision;
    }

    public AggregatedCondition getReady() {
        return ready;
    }

    public AggregatedCondition getInProgress() {
        return inProgress;
    }

    public AggregatedCondition getError() {
        return error;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DynamicResourceRevisionStatus that = (DynamicResourceRevisionStatus) o;
        return revision == that.revision &&
                Objects.equals(ready, that.ready) &&
                Objects.equals(inProgress, that.inProgress) &&
                Objects.equals(error, that.error);
    }

    @Override
    public int hashCode() {
        return Objects.hash(revision, ready, inProgress, error);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("revision", revision)
                .add("ready", ready)
                .add("inProgress", inProgress)
                .add("error", error)
                .toString();
    }

    private static Optional<Condition> mergeConditions(Optional<Condition> first, Optional<Condition> second,
                                                       BiFunction<Condition, Condition, Condition> reducer) {
        if (first.isEmpty()) {
            return second;
        }
        if (second.isEmpty()) {
            return first;
        }

        return Optional.of(reducer.apply(first.get(), second.get()));
    }

    private static AggregatedCondition conjunctAggregatedConditions(AggregatedCondition c1, AggregatedCondition c2) {
        return new AggregatedCondition(c1.getPodCount() + c2.getPodCount(),
                mergeConditions(c1.getCondition(), c2.getCondition(), (a, b) -> a.isTrue() ? b : a));
    }

    private static AggregatedCondition disjunctAggregatedConditions(AggregatedCondition c1, AggregatedCondition c2) {
        return new AggregatedCondition(c1.getPodCount() + c2.getPodCount(),
                mergeConditions(c1.getCondition(), c2.getCondition(), (a, b) -> a.isTrue() ? a : b));
    }

    public static DynamicResourceRevisionStatus emptyStatus() {
        return new DynamicResourceRevisionStatus(0, new AggregatedCondition(0, Optional.empty()),
                new AggregatedCondition(0, Optional.empty()), new AggregatedCondition(0, Optional.empty()));
    }

    public static DynamicResourceRevisionStatus aggregateFromClusters(Map<String, DynamicResourceRevisionStatus> perClusterStatus) {
        long lastRevision = perClusterStatus.values().stream()
                .map(DynamicResourceRevisionStatus::getRevision).reduce(Math::max).orElse(0L);

        AggregatedCondition empty = new AggregatedCondition(0, Optional.empty());

        List<DynamicResourceRevisionStatus> statuses =
                perClusterStatus.values().stream().filter(r -> r.getRevision() == lastRevision).collect(Collectors.toList());

        return new DynamicResourceRevisionStatus(lastRevision,
                statuses.stream().map(DynamicResourceRevisionStatus::getReady).reduce(empty,
                        DynamicResourceRevisionStatus::conjunctAggregatedConditions),
                statuses.stream().map(DynamicResourceRevisionStatus::getInProgress).reduce(empty,
                        DynamicResourceRevisionStatus::disjunctAggregatedConditions),
                statuses.stream().map(DynamicResourceRevisionStatus::getError).reduce(empty,
                        DynamicResourceRevisionStatus::disjunctAggregatedConditions)
        );
    }

    @Override
    public boolean isReady() {
        return true;
    }
}
