package ru.yandex.infra.stage.dto;

import java.util.Optional;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

public class LogbrokerConfig {
    private final Optional<DownloadableResource> logbrokerAgentLayer;
    private final SidecarBringupMode sidecarBringupMode;
    private final Optional<String> logsVirtualDiskIdRef;
    private final Optional<SidecarVolumeSettings> sidecarVolumeSettings;
    private final LogbrokerDestroyPolicy destroyPolicy;
    private final LogbrokerTopicRequest topicRequest;
    private final Optional<AllComputeResources> podAdditionalResourcesRequest;

    //logsVirtualDiskIdRef - user disk id for volume with porto log files
    public LogbrokerConfig(Optional<DownloadableResource> logbrokerAgentLayer,
                           SidecarBringupMode sidecarBringupMode,
                           Optional<String> logsVirtualDiskIdRef,
                           Optional<SidecarVolumeSettings> sidecarVolumeSettings,
                           LogbrokerDestroyPolicy destroyPolicy,
                           LogbrokerTopicRequest topicRequest,
                           Optional<AllComputeResources> podAdditionalResourcesRequest) {
        this.logbrokerAgentLayer = logbrokerAgentLayer;
        this.sidecarBringupMode = sidecarBringupMode;
        this.logsVirtualDiskIdRef = logsVirtualDiskIdRef;
        this.sidecarVolumeSettings = sidecarVolumeSettings;
        this.destroyPolicy = destroyPolicy;
        this.topicRequest = topicRequest;
        this.podAdditionalResourcesRequest = podAdditionalResourcesRequest;
    }

    public Optional<DownloadableResource> getLogbrokerAgentLayer() {
        return logbrokerAgentLayer;
    }

    public SidecarBringupMode getSidecarBringupMode() {
        return sidecarBringupMode;
    }

    public boolean bringupSidecarMandatory() {
        return sidecarBringupMode == SidecarBringupMode.MANDATORY;
    }

    public Optional<String> getLogsVirtualDiskIdRef() {
        return logsVirtualDiskIdRef;
    }

    public Optional<SidecarVolumeSettings> getSidecarVolumeSettings() {
        return sidecarVolumeSettings;
    }

    public LogbrokerDestroyPolicy getDestroyPolicy() {
        return destroyPolicy;
    }

    public LogbrokerTopicRequest getTopicRequest() {
        return topicRequest;
    }

    public Optional<AllComputeResources> getPodAdditionalResourcesRequest() {
        return podAdditionalResourcesRequest;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        LogbrokerConfig that = (LogbrokerConfig) o;
        return Objects.equal(logbrokerAgentLayer, that.logbrokerAgentLayer)
                && sidecarBringupMode == that.sidecarBringupMode
                && Objects.equal(logsVirtualDiskIdRef, that.logsVirtualDiskIdRef)
                && Objects.equal(sidecarVolumeSettings, that.sidecarVolumeSettings)
                && Objects.equal(destroyPolicy, that.destroyPolicy)
                && Objects.equal(topicRequest, that.topicRequest)
                && Objects.equal(podAdditionalResourcesRequest, that.podAdditionalResourcesRequest);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                logbrokerAgentLayer,
                sidecarBringupMode,
                logsVirtualDiskIdRef,
                sidecarVolumeSettings,
                destroyPolicy,
                topicRequest,
                podAdditionalResourcesRequest
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("logbrokerAgentLayer", logbrokerAgentLayer)
                .add("sidecarBringupMode", sidecarBringupMode)
                .add("logsVirtualDiskIdRef", logsVirtualDiskIdRef)
                .add("sidecarVolumeSettings", sidecarVolumeSettings)
                .add("destroyPolicy", destroyPolicy)
                .add("topicRequest", topicRequest)
                .add("podAdditionalResourcesRequest", podAdditionalResourcesRequest)
                .toString();
    }

    public enum SidecarBringupMode {
        UNRECOGNIZED,
        UNKNOWN,
        DEFAULT,
        MANDATORY
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static final class Builder {

        private Optional<DownloadableResource> logbrokerAgentLayer;
        private SidecarBringupMode sidecarBringupMode;
        private Optional<String> logsVirtualDiskIdRef;
        private Optional<SidecarVolumeSettings> sidecarVolumeSettings;
        private LogbrokerDestroyPolicy destroyPolicy;
        private LogbrokerTopicRequest topicRequest;
        private Optional<AllComputeResources> podAdditionalResourcesRequest;

        private Builder(LogbrokerConfig logbrokerConfig) {
            this.logbrokerAgentLayer = logbrokerConfig.logbrokerAgentLayer;
            this.sidecarBringupMode = logbrokerConfig.sidecarBringupMode;
            this.logsVirtualDiskIdRef = logbrokerConfig.logsVirtualDiskIdRef;
            this.sidecarVolumeSettings = logbrokerConfig.sidecarVolumeSettings;
            this.destroyPolicy = logbrokerConfig.destroyPolicy;
            this.topicRequest = logbrokerConfig.topicRequest;
            this.podAdditionalResourcesRequest = logbrokerConfig.podAdditionalResourcesRequest;
        }

        public Builder withSidecarBringupMode(SidecarBringupMode sidecarBringupMode) {
            this.sidecarBringupMode = sidecarBringupMode;
            return this;
        }

        public Builder withLogsVirtualDiskIdRef(Optional<String> logsVirtualDiskIdRef) {
            this.logsVirtualDiskIdRef = logsVirtualDiskIdRef;
            return this;
        }

        public Builder withLogbrokerAgentLayer(Optional<DownloadableResource> logbrokerAgentLayer) {
            this.logbrokerAgentLayer = logbrokerAgentLayer;
            return this;
        }

        public Builder withSidecarVolumeSettings(Optional<SidecarVolumeSettings> sidecarVolumeSettings) {
            this.sidecarVolumeSettings = sidecarVolumeSettings;
            return this;
        }

        public Builder withDestroyPolicy(LogbrokerDestroyPolicy destroyPolicy) {
            this.destroyPolicy = destroyPolicy;
            return this;
        }

        public Builder withTopicRequest(LogbrokerTopicRequest topicRequest) {
            this.topicRequest = topicRequest;
            return this;
        }

        public Builder withPodAdditionalResourcesRequest(Optional<AllComputeResources> podAdditionalResourcesRequest) {
            this.podAdditionalResourcesRequest = podAdditionalResourcesRequest;
            return this;
        }

        public LogbrokerConfig build() {
            return new LogbrokerConfig(
                    logbrokerAgentLayer,
                    sidecarBringupMode,
                    logsVirtualDiskIdRef,
                    sidecarVolumeSettings,
                    destroyPolicy,
                    topicRequest,
                    podAdditionalResourcesRequest
            );
        }
    }
}
