package ru.yandex.infra.stage.dto;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import one.util.streamex.StreamEx;

import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.yp.client.api.DataModel.TPodSpec;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetScaleSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yt.ytree.TAttributeDictionary;

import static java.util.Optional.empty;

public class McrsUnitSpec extends DeployUnitSpecDetails {
    private final TMultiClusterReplicaSetSpec spec;
    private final Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor;

    public McrsUnitSpec(TMultiClusterReplicaSetSpec spec, Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor) {
        this.spec = spec;
        this.podAgentConfigExtractor = podAgentConfigExtractor;
    }

    public TMultiClusterReplicaSetSpec getSpec() {
        return spec;
    }

    @Override
    public Set<String> extractClusters() {
        return spec.getClustersList().stream()
                .map(TMultiClusterReplicaSetSpec.TClusterReplicaSetSpecPreferences::getCluster)
                .collect(Collectors.toSet());
    }

    @Override
    public TPodSpec getPodSpec() {
        return spec.getPodTemplateSpec().getSpec();
    }

    @Override
    public TAttributeDictionary getLabels() {
        return spec.getPodTemplateSpec().getLabels();
    }

    @Override
    public DeployUnitStatusDetails createStatusDetails(String deployPrimitiveId, List<String> endpointSetIds, Map<String,
            Optional<TReplicaSetStatus>> statuses) {
        return new McrsUnitStatus(deployPrimitiveId, StreamEx.of(extractClusters())
                .toMap(cluster -> new McrsUnitStatus.PerClusterStatus(endpointSetIds)));
    }

    @Override
    public Retainment shouldRetain(ClusterAndType clusterAndType) {
        if (clusterAndType.equals(ClusterAndType.mcrs())) {
            return new Retainment(true, "MCRS spec present");
        } else if (clusterAndType.getType() == YpObjectType.ENDPOINT_SET) {
            return retainByCluster(clusterAndType);
        } else {
            return retainUnknownType(clusterAndType.getType());
        }
    }

    @Override
    public String objectDescription() {
        return "MCRS";
    }

    @Override
    public void validateDeploymentStrategy(List<String> errors) {
        if (!spec.hasDeploymentStrategy()) {
            errors.add("Empty deployment strategy for mcrs");
        }
    }

    @Override
    public boolean supportsAutoScaling() {
        return false;
    }

    @Override
    public Optional<TReplicaSetScaleSpec> getAutoscale(String cluster) {
        return empty();
    }

    @Override
    public PodAgentConfig extractPodAgentConfig() { return podAgentConfigExtractor.apply(spec.getPodTemplateSpec().getSpec().getPodAgentPayload().getMeta()); }

    @VisibleForTesting
    public McrsUnitSpec withPodAgentConfigExtractor(Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor) {
        return new McrsUnitSpec(spec, podAgentConfigExtractor);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof McrsUnitSpec)) return false;
        McrsUnitSpec that = (McrsUnitSpec) o;
        return Objects.equals(spec, that.spec);
    }

    @Override
    public int hashCode() {
        return Objects.hash(spec);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("spec", spec)
                .toString();
    }
}
