package ru.yandex.infra.stage.dto;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.MoreObjects;

public class NetworkDefaults {
    private final String networkId;
    private final boolean overrideIp6AddressRequests;
    private final boolean overrideIp6SubnetRequests;
    private final Optional<String> ip4AddressPoolId;
    private final List<String> virtualServiceIds;

    public NetworkDefaults(String networkId,
                           boolean overrideIp6AddressRequests,
                           boolean overrideIp6SubnetRequests,
                           Optional<String> ip4AddressPoolId,
                           List<String> virtualServiceIds) {
        this.networkId = networkId;
        this.overrideIp6AddressRequests = overrideIp6AddressRequests;
        this.overrideIp6SubnetRequests = overrideIp6SubnetRequests;
        this.ip4AddressPoolId = ip4AddressPoolId;
        this.virtualServiceIds = virtualServiceIds;
    }

    public String getNetworkId() {
        return networkId;
    }

    public boolean isOverrideIp6AddressRequests() {
        return overrideIp6AddressRequests;
    }

    public boolean isOverrideIp6SubnetRequests() {
        return overrideIp6SubnetRequests;
    }

    public Optional<String> getIp4AddressPoolId() {
        return ip4AddressPoolId;
    }

    public List<String> getVirtualServiceIds() {
        return virtualServiceIds;
    }

    public NetworkDefaults withNetworkId(String networkId) {
        return toBuilder().withNetworkId(networkId).build();
    }

    public NetworkDefaults withOverrideIp6AddressRequests(boolean overrideIp6AddressRequests) {
        return toBuilder().withOverrideIp6AddressRequests(overrideIp6AddressRequests).build();
    }

    public NetworkDefaults withIp4AddressPoolId(String ip4AddressPoolId) {
        return toBuilder().withIp4AddressPoolId(ip4AddressPoolId).build();
    }

    public NetworkDefaults withVirtualServiceIds(List<String> virtualServiceIds) {
        return toBuilder().withVirtualServiceIds(virtualServiceIds).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NetworkDefaults that = (NetworkDefaults) o;
        return overrideIp6AddressRequests == that.overrideIp6AddressRequests &&
                overrideIp6SubnetRequests == that.overrideIp6SubnetRequests &&
                Objects.equals(networkId, that.networkId) &&
                Objects.equals(ip4AddressPoolId, that.ip4AddressPoolId) &&
                Objects.equals(virtualServiceIds, that.virtualServiceIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                networkId,
                overrideIp6AddressRequests,
                overrideIp6SubnetRequests,
                ip4AddressPoolId,
                virtualServiceIds
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("networkId", networkId)
                .add("overrideIp6AddressRequests", overrideIp6AddressRequests)
                .add("overrideIp6SubnetRequests", overrideIp6SubnetRequests)
                .add("ip4AddressPoolId", ip4AddressPoolId)
                .add("virtualServiceIds", virtualServiceIds)
                .toString();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private String networkId;
        private boolean overrideIp6AddressRequests;
        private boolean overrideIp6SubnetRequests;
        private Optional<String> ip4AddressPoolId;
        private List<String> virtualServiceIds;

        Builder(NetworkDefaults networkDefaults) {
            this.networkId = networkDefaults.networkId;
            this.overrideIp6AddressRequests = networkDefaults.overrideIp6AddressRequests;
            this.overrideIp6SubnetRequests = networkDefaults.overrideIp6SubnetRequests;
            this.ip4AddressPoolId = networkDefaults.ip4AddressPoolId;
            this.virtualServiceIds = networkDefaults.virtualServiceIds;
        }

        public NetworkDefaults build() {
            return new NetworkDefaults(
                    networkId,
                    overrideIp6AddressRequests,
                    overrideIp6SubnetRequests,
                    ip4AddressPoolId,
                    virtualServiceIds
            );
        }

        public Builder withNetworkId(String networkId) {
            this.networkId = networkId;
            return this;
        }

        public Builder withOverrideIp6AddressRequests(boolean overrideIp6AddressRequests) {
            this.overrideIp6AddressRequests = overrideIp6AddressRequests;
            return this;
        }

        public Builder withOverrideIp6SubnetRequests(boolean overrideIp6SubnetRequests) {
            this.overrideIp6SubnetRequests = overrideIp6SubnetRequests;
            return this;
        }

        private Builder withIp4AddressPoolId(Optional<String> ip4AddressPoolId) {
            this.ip4AddressPoolId = ip4AddressPoolId;
            return this;
        }

        Builder withIp4AddressPoolId(String ip4AddressPoolId) {
            return withIp4AddressPoolId(Optional.ofNullable(ip4AddressPoolId));
        }

        public Builder withVirtualServiceIds(List<String> virtualServiceIds) {
            this.virtualServiceIds = virtualServiceIds;
            return this;
        }
    }
}
