package ru.yandex.infra.stage.dto;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Either;
import ru.yandex.infra.stage.yp.Retainment;
import ru.yandex.yp.client.api.DataModel.TPodSpec;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TReplicaSetScaleSpec;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yt.ytree.TAttributeDictionary;

public class ReplicaSetUnitSpec extends DeployUnitSpecDetails {
    public static class PerClusterSettings {
        private final Either<Integer, TReplicaSetScaleSpec> podCountPolicy;
        private final Optional<ReplicaSetDeploymentStrategy> deploymentStrategy;

        public PerClusterSettings(Either<Integer, TReplicaSetScaleSpec> podCountPolicy,
                                  Optional<ReplicaSetDeploymentStrategy> deploymentStrategy) {
            this.deploymentStrategy = deploymentStrategy;
            this.podCountPolicy = podCountPolicy;

        }
        public PerClusterSettings(Either<Integer, TReplicaSetScaleSpec> podCountPolicy,
                                  Optional<ReplicaSetDeploymentStrategy> deploymentStrategy,
                                  Optional<TReplicaSetStatus> rawStatus) {
            this.deploymentStrategy = deploymentStrategy;
            this.podCountPolicy = podCountPolicy;
        }

        public Optional<Integer> getPodCount() {
            return podCountPolicy.leftO().toOptional();
        }

        public boolean hasPodCount() {
            return podCountPolicy.isLeft();
        }

        public Optional<ReplicaSetDeploymentStrategy> getDeploymentStrategy() {
            return deploymentStrategy;
        }

        public Optional<TReplicaSetScaleSpec> getAutoscale() {
            return podCountPolicy.rightO().toOptional();
        }

        public boolean hasAutoscale() {
            return podCountPolicy.isRight();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof PerClusterSettings)) {
                return false;
            }
            PerClusterSettings that = (PerClusterSettings) o;
            return Objects.equals(podCountPolicy, that.podCountPolicy) &&
                    Objects.equals(deploymentStrategy, that.deploymentStrategy);
        }

        @Override
        public int hashCode() {
            return Objects.hash(podCountPolicy, deploymentStrategy);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("podCountPolicy", podCountPolicy)
                    .add("deploymentStrategy", deploymentStrategy)
                    .toString();
        }
    }

    private final Map<String, PerClusterSettings> perClusterSettings;
    private final TReplicaSetSpec specTemplate;
    private final Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor;

    public ReplicaSetUnitSpec(TReplicaSetSpec specTemplate, Map<String, PerClusterSettings> perClusterSettings,
                              Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor) {
        this.perClusterSettings = perClusterSettings;
        this.specTemplate = specTemplate;
        this.podAgentConfigExtractor = podAgentConfigExtractor;
    }

    public Map<String, PerClusterSettings> getPerClusterSettings() {
        return perClusterSettings;
    }

    public PerClusterSettings getSettingsForCluster(String cluster) {
        return perClusterSettings.get(cluster);
    }

    public TReplicaSetSpec getSpecTemplate() {
        return specTemplate;
    }

    @Override
    public Set<String> extractClusters() {
        return perClusterSettings.keySet();
    }

    @Override
    public TPodSpec getPodSpec() {
        return specTemplate.getPodTemplateSpec().getSpec();
    }

    @Override
    public TAttributeDictionary getLabels() {
        return specTemplate.getPodTemplateSpec().getLabels();
    }

    public DeployUnitStatusDetails createStatusDetails(String deployPrimitiveId, List<String> endpointSetIds, Map<String, Optional<TReplicaSetStatus>> statuses) {
        return new ReplicaSetUnitStatus(StreamEx.of(extractClusters())
                .toMap(cluster ->
                        new ReplicaSetUnitStatus.PerClusterStatus(deployPrimitiveId, endpointSetIds,
                                perClusterSettings.get(cluster).hasAutoscale() ? deployPrimitiveId : "",
                                statuses.getOrDefault(cluster, Optional.empty()))));
    }

    @Override
    public Retainment shouldRetain(ClusterAndType clusterAndType) {
        if (clusterAndType.getType() == YpObjectType.REPLICA_SET || clusterAndType.getType() == YpObjectType.ENDPOINT_SET) {
            return retainByCluster(clusterAndType);
        } else if (clusterAndType.getType() == YpObjectType.HORIZONTAL_POD_AUTOSCALER) {
            PerClusterSettings settings = perClusterSettings.get(clusterAndType.getCluster().orElseThrow());
            if (settings != null && settings.hasAutoscale()) {
                return new Retainment(true,
                        String.format("%s has %s in cluster '%s'",
                                objectDescription(),
                                clusterAndType.getType(),
                                clusterAndType.getCluster().get()));
            } else {
                return new Retainment(false,
                        String.format("%s does not have %s in cluster '%s'",
                                objectDescription(),
                                clusterAndType.getType(),
                                clusterAndType.getCluster().get()));
            }
        }
        return retainUnknownType(clusterAndType.getType());
    }

    @Override
    public String objectDescription() {
        return "Replica set";
    }

    @Override
    public void validateDeploymentStrategy(List<String> errors) {
        if (specTemplate.hasDeploymentStrategy()) {
            return;
        }
        perClusterSettings.forEach((cluster, settings) -> {
            if (settings.getDeploymentStrategy().isEmpty()) {
                errors.add(String.format("Empty deployment strategy for replica set on '%s'", cluster));
            }
        });
    }

    @Override
    public boolean supportsAutoScaling() {
        return true;
    }

    @Override
    public Optional<TReplicaSetScaleSpec> getAutoscale(String cluster) {
        return perClusterSettings.get(cluster).getAutoscale();
    }

    @Override
    public PodAgentConfig extractPodAgentConfig() {
        return podAgentConfigExtractor.apply(specTemplate.getPodTemplateSpec().getSpec().getPodAgentPayload().getMeta());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ReplicaSetUnitSpec)) {
            return false;
        }
        ReplicaSetUnitSpec that = (ReplicaSetUnitSpec) o;
        return Objects.equals(perClusterSettings, that.perClusterSettings) &&
                Objects.equals(specTemplate, that.specTemplate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(perClusterSettings, specTemplate);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("perClusterSettings", perClusterSettings)
                .add("spec", specTemplate)
                .toString();
    }

    private ReplicaSetUnitSpec withSpecTemplate(TReplicaSetSpec specTemplate) {
        return new ReplicaSetUnitSpec(
                specTemplate,
                perClusterSettings,
                podAgentConfigExtractor
        );
    }

    private ReplicaSetUnitSpec withPodTemplateSpec(TPodTemplateSpec podSpecTemplate) {
        return withSpecTemplate(
                specTemplate.toBuilder()
                        .setPodTemplateSpec(podSpecTemplate)
                        .build()
        );
    }

    @VisibleForTesting
    public ReplicaSetUnitSpec withPodSpec(TPodSpec podSpec) {
        return withPodTemplateSpec(
                specTemplate.getPodTemplateSpec().toBuilder()
                        .setSpec(podSpec)
                        .build()
        );
    }

    @VisibleForTesting
    public ReplicaSetUnitSpec withPodAgentConfigExtractor(
            Function<TPodSpec.TPodAgentDeploymentMeta, PodAgentConfig> podAgentConfigExtractor
    ) {
        return new ReplicaSetUnitSpec(
                specTemplate,
                perClusterSettings,
                podAgentConfigExtractor
        );
    }
}
