package ru.yandex.infra.stage.dto;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;

import static ru.yandex.infra.stage.util.JsonUtils.DEFAULT_MAPPER;

public class TvmConfig {
    public static final int DEFAULT_CLIENT_PORT = 2;
    public static final int DEFAULT_MONITORING_PORT = 12510;
    public static final int DEFAULT_SOLOMON_TVM_ID = 2012028;
    public static final int DEFAULT_STAGE_TVM_ID = 2020703;
    public static final int DEFAULT_CPU_LIMIT = 100;
    public static final int DEFAULT_MEMORY_LIMIT_MB = 30;

    private final Mode mode;
    private final String blackboxEnvironment;
    private final List<TvmClient> clients;
    private final int clientPort;
    private final int cpuLimit;
    private final int memoryLimitMb;
    private final Optional<DownloadableResource> tvmtoolLayer;
    private final int stageTvmId;
    private final int solomonTvmId;
    private final int monitoringPort;
    private final Optional<SidecarVolumeSettings> sidecarVolumeSettings;
    private final boolean useSystemCerts;


    public TvmConfig(Mode mode,
                     String blackboxEnvironment,
                     List<TvmClient> clients,
                     OptionalInt clientPort,
                     OptionalInt cpuLimit, OptionalInt memoryLimitMb, Optional<DownloadableResource> tvmtoolLayer,
                     OptionalInt stageTvmId, OptionalInt solomonTvmId, OptionalInt monitoringPort,
                     Optional<SidecarVolumeSettings> sidecarVolumeSettings, boolean useSystemCerts) {
        this.mode = mode;
        this.blackboxEnvironment = blackboxEnvironment;
        this.clients = clients;
        this.clientPort = clientPort.orElse(DEFAULT_CLIENT_PORT);
        this.cpuLimit = cpuLimit.orElse(DEFAULT_CPU_LIMIT);
        this.memoryLimitMb = memoryLimitMb.orElse(DEFAULT_MEMORY_LIMIT_MB);
        this.tvmtoolLayer = tvmtoolLayer;
        this.stageTvmId = stageTvmId.orElse(DEFAULT_STAGE_TVM_ID);
        this.solomonTvmId = solomonTvmId.orElse(DEFAULT_SOLOMON_TVM_ID);
        this.monitoringPort = monitoringPort.orElse(DEFAULT_MONITORING_PORT);
        this.sidecarVolumeSettings = sidecarVolumeSettings;
        this.useSystemCerts = useSystemCerts;
    }

    public int getCpuLimit() {
        return cpuLimit;
    }

    public int getMemoryLimitMb() {
        return memoryLimitMb;
    }

    public boolean isEnabled() {
        return mode == Mode.ENABLED;
    }

    public Mode getMode() {
        return mode;
    }

    public String getBlackboxEnvironment() {
        return blackboxEnvironment;
    }

    public int getBlackboxEnvironmentValue(Map<String, Integer> blackboxEnvironments) {
        return blackboxEnvironments.get(blackboxEnvironment);
    }

    public List<TvmClient> getClients() {
        return clients;
    }

    public int getClientPort() {
        return clientPort;
    }

    public Optional<DownloadableResource> getTvmtoolLayer() {
        return tvmtoolLayer;
    }

    public int getStageTvmId() {
        return stageTvmId;
    }

    public int getSolomonTvmId() {
        return solomonTvmId;
    }

    public int getMonitoringPort() {
        return monitoringPort;
    }

    public Optional<SidecarVolumeSettings> getSidecarVolumeSettings() { return sidecarVolumeSettings; }

    public boolean getUseSystemCerts() {
        return useSystemCerts;
    }

    @VisibleForTesting
    public TvmConfig withClients(List<TvmClient> clients) {
         return new TvmConfig(mode, blackboxEnvironment, clients,
                 OptionalInt.of(clientPort), OptionalInt.of(cpuLimit),
                 OptionalInt.of(memoryLimitMb), tvmtoolLayer, OptionalInt.of(stageTvmId),
                 OptionalInt.of(solomonTvmId), OptionalInt.of(monitoringPort), sidecarVolumeSettings, useSystemCerts);
    }

    @VisibleForTesting
    public TvmConfig withMode(Mode mode) {
        return new TvmConfig(mode, blackboxEnvironment, clients,
                OptionalInt.of(clientPort), OptionalInt.of(cpuLimit),
                OptionalInt.of(memoryLimitMb), tvmtoolLayer, OptionalInt.of(stageTvmId),
                OptionalInt.of(solomonTvmId), OptionalInt.of(monitoringPort), sidecarVolumeSettings, useSystemCerts);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        TvmConfig tvmConfig = (TvmConfig) o;
        return clientPort == tvmConfig.clientPort &&
                cpuLimit == tvmConfig.cpuLimit &&
                memoryLimitMb == tvmConfig.memoryLimitMb &&
                stageTvmId == tvmConfig.stageTvmId &&
                solomonTvmId == tvmConfig.solomonTvmId &&
                monitoringPort == tvmConfig.monitoringPort &&
                mode == tvmConfig.mode &&
                Objects.equals(blackboxEnvironment, tvmConfig.blackboxEnvironment) &&
                Objects.equals(clients, tvmConfig.clients) &&
                Objects.equals(tvmtoolLayer, tvmConfig.tvmtoolLayer) &&
                Objects.equals(sidecarVolumeSettings, tvmConfig.sidecarVolumeSettings) &&
                Objects.equals(useSystemCerts, tvmConfig.useSystemCerts);
    }

    @Override
    public int hashCode() {
        return Objects.hash(mode, blackboxEnvironment, clients, clientPort, cpuLimit, memoryLimitMb, tvmtoolLayer, stageTvmId, solomonTvmId, monitoringPort, sidecarVolumeSettings, useSystemCerts);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("mode", mode)
                .add("blackboxEnvironment", blackboxEnvironment)
                .add("clients", clients)
                .add("clientPort", clientPort)
                .add("cpuLimit", cpuLimit)
                .add("memoryLimitMb", memoryLimitMb)
                .add("tvmtoolLayer", tvmtoolLayer)
                .add("stageTvmId", stageTvmId)
                .add("solomonTvmId", solomonTvmId)
                .add("monitoringPort", monitoringPort)
                .add("sidecarVolumeSettings", sidecarVolumeSettings)
                .add("useSystemCerts", useSystemCerts)
                .toString();
    }

    // Generate config for tvmtool. See
    // https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/ for details.
    public String toJsonString(Map<String, Integer> blackboxEnvironments) {
        ObjectMapper mapper = DEFAULT_MAPPER;
        ObjectNode confNode = mapper.createObjectNode();
        confNode.put("BbEnvType", getBlackboxEnvironmentValue(blackboxEnvironments));
        if (getUseSystemCerts()) {
            confNode.put("use_system_certs", getUseSystemCerts());
        }
        confNode.put("port", clientPort);
        ObjectNode clientsNode = mapper.createObjectNode();
        clients.forEach(client -> {
            ObjectNode clientNode = mapper.createObjectNode();
            TvmApp src = client.getSource();
            clientNode.put("self_tvm_id", src.getAppId());
            List<TvmApp> dsts = client.getDestinations();
            if(!dsts.isEmpty()) {
                clientNode.put("secret", String.format("env:%s", client.getTvmClientSecretEnvName()));
                if (!client.getRolesForIdmSlug().isEmpty()) {
                    clientNode.put("roles_for_idm_slug", client.getRolesForIdmSlug());
                }
                ObjectNode dstsNode = mapper.createObjectNode();
                dsts.forEach(dst -> {
                    ObjectNode dstNode = mapper.createObjectNode();
                    dstNode.put("dst_id", dst.getAppId());
                    dstsNode.set(dst.getAliasOrAppId(), dstNode);
                });
                clientNode.set("dsts", dstsNode);
            }

            clientsNode.set(src.getAliasOrAppId(), clientNode);
        });
        confNode.set("clients", clientsNode);
        ObjectNode solomonNode = mapper.createObjectNode();
        solomonNode.put("stage_tvm_id", stageTvmId);
        solomonNode.put("solomon_tvm_id", solomonTvmId);
        solomonNode.put("port", monitoringPort);
        confNode.set("solomon", solomonNode);
        try {
            return mapper.writeValueAsString(confNode);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public enum Mode {
        UNRECOGNIZED,
        UNKNOWN,
        DISABLED,
        ENABLED
    }
}
