package ru.yandex.infra.stage.dto.datamodel;

import java.util.Objects;

import com.google.common.base.MoreObjects;

public class AntiaffinityConstraint {
    private static final String NODE_STRING = "node";
    private static final String RACK_STRING = "rack";
    private static final String DC_STRING = "dc";

    public enum Zone {
        NODE(NODE_STRING),
        RACK(RACK_STRING),
        DC(DC_STRING);

        private final String ypName;

        Zone(String ypName) {
            this.ypName = ypName;
        }

        public String getYpName() {
            return ypName;
        }

        public static Zone fromYpName(String ypName) {
            switch (ypName) {
                case NODE_STRING:
                    return NODE;
                case RACK_STRING:
                    return RACK;
                case DC_STRING:
                    return DC;
                default:
                    throw new IllegalArgumentException(String.format("Unknown antiaffinity zone %s", ypName));
            }
        }
    }

    private final Zone zone;
    private final long maxPods;

    public AntiaffinityConstraint(Zone zone, long maxPods) {
        this.zone = zone;
        this.maxPods = maxPods;
    }

    public Zone getZone() {
        return zone;
    }

    public long getMaxPods() {
        return maxPods;
    }

    public static AntiaffinityConstraint node(long maxPods) {
        return new AntiaffinityConstraint(Zone.NODE, maxPods);
    }

    public static AntiaffinityConstraint rack(long maxPods) {
        return new AntiaffinityConstraint(Zone.RACK, maxPods);
    }

    public static AntiaffinityConstraint dc(long maxPods) {
        return new AntiaffinityConstraint(Zone.DC, maxPods);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof AntiaffinityConstraint)) return false;
        AntiaffinityConstraint that = (AntiaffinityConstraint) o;
        return maxPods == that.maxPods &&
                zone == that.zone;
    }

    @Override
    public int hashCode() {
        return Objects.hash(zone, maxPods);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("zone", zone)
                .add("maxPods", maxPods)
                .toString();
    }
}
