package ru.yandex.infra.stage.inject;

import java.time.Clock;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.protobuf.Message;
import one.util.streamex.StreamEx;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.stage.AclFilter;
import ru.yandex.infra.stage.DeployPrimitiveControllerFactory;
import ru.yandex.infra.stage.DeployUnitControllerFactory;
import ru.yandex.infra.stage.GlobalContext;
import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.StageControllerFactory;
import ru.yandex.infra.stage.StageControllerImpl;
import ru.yandex.infra.stage.StageStatusSender;
import ru.yandex.infra.stage.StageValidator;
import ru.yandex.infra.stage.ValidationController;
import ru.yandex.infra.stage.ValidationControllerFactory;
import ru.yandex.infra.stage.ValidationControllerImpl;
import ru.yandex.infra.stage.deployunit.DeployUnitController;
import ru.yandex.infra.stage.deployunit.DeployUnitControllerImpl;
import ru.yandex.infra.stage.deployunit.DynamicResourceController;
import ru.yandex.infra.stage.deployunit.EndpointSetController;
import ru.yandex.infra.stage.deployunit.HorizontalPodAutoscalerController;
import ru.yandex.infra.stage.deployunit.LogbrokerTopicConfigResolver;
import ru.yandex.infra.stage.deployunit.MultiClusterReplicaSetController;
import ru.yandex.infra.stage.deployunit.ObjectController;
import ru.yandex.infra.stage.deployunit.ReadinessStatus;
import ru.yandex.infra.stage.deployunit.SandboxResourcesResolver;
import ru.yandex.infra.stage.deployunit.SingleReplicaSetController;
import ru.yandex.infra.stage.docker.DockerImagesResolver;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitSpecDetails;
import ru.yandex.infra.stage.dto.DeployUnitStatus;
import ru.yandex.infra.stage.dto.DynamicResourceMeta;
import ru.yandex.infra.stage.dto.DynamicResourceRevisionStatus;
import ru.yandex.infra.stage.dto.DynamicResourceStatus;
import ru.yandex.infra.stage.dto.HorizontalPodAutoscalerMeta;
import ru.yandex.infra.stage.dto.McrsUnitSpec;
import ru.yandex.infra.stage.dto.ReplicaSetUnitSpec;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.primitives.DeployPrimitiveController;
import ru.yandex.infra.stage.primitives.DeployPrimitiveStatus;
import ru.yandex.infra.stage.primitives.McrsDeployPrimitiveController;
import ru.yandex.infra.stage.primitives.ReplicaSetDeployPrimitiveController;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.AclUpdater;
import ru.yandex.infra.stage.yp.AsyncYpClientsMap;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManager;
import ru.yandex.infra.stage.yp.RelationController;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceSpec;
import ru.yandex.yp.client.api.DynamicResource.TDynamicResourceStatus;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerSpec;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerStatus;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetStatus;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.model.YpObjectType;

public class ControllerFactoryImpl implements StageControllerFactory, DeployUnitControllerFactory,
        DeployPrimitiveControllerFactory,
        ValidationControllerFactory {

    private final Clock clock;
    private final Map<String, ObjectLifeCycleManager<SchemaMeta, DataModel.TEndpointSetSpec,
                DataModel.TEndpointSetStatus>> endpointSetManagers;
    private final Map<String, ObjectLifeCycleManager<HorizontalPodAutoscalerMeta, THorizontalPodAutoscalerSpec,
            THorizontalPodAutoscalerStatus>> autoscalerManagers;
    private final Map<String, ObjectLifeCycleManager<SchemaMeta, TReplicaSetSpec, TReplicaSetStatus>> replicaSetManagers;
    private final ObjectLifeCycleManager<SchemaMeta, TMultiClusterReplicaSetSpec, TMultiClusterReplicaSetStatus> mcrsManager;
    private final Map<String, ObjectLifeCycleManager<DynamicResourceMeta, TDynamicResourceSpec, TDynamicResourceStatus>> dynamicResourceManagers;
    private final DockerImagesResolver dockerImagesResolver;
    private final Converter converter;
    private final StageValidator validator;
    private final StageStatusSender stageStatusSender;
    private final AclUpdater commonAclUpdater;
    private final AclUpdater replicaSetAclUpdater;
    private final AclUpdater horizontalPodAutoscalerAclUpdater;
    private final LogbrokerTopicConfigResolver logbrokerTopicConfigResolver;
    private final SandboxResourcesResolver sandboxResolver;
    private final AclFilter aclFilter;
    private final RelationController relationController;
    private final GlobalContext globalContext;
    private final SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher;
    private final SpecPatcher<DataModel.TEndpointSetSpec.Builder> endpointSetSpecPatcher;
    private final SpecPatcher<TDynamicResourceSpec.Builder> dynamicResourceSpecPatcher;


    public ControllerFactoryImpl(AsyncYpClientsMap ypClients,
                                 Clock clock,
                                 AclUpdater commonAclUpdater,
                                 DockerImagesResolver dockerImagesResolver, Converter converter,
                                 StageValidator validator, StageStatusSender stageStatusSender,
                                 ObjectLifeCycleManagerFactory facadeFactory, AclUpdater replicaSetAclUpdater,
                                 AclUpdater horizontalPodAutoscalerAclUpdater, AclFilter aclFilter,
                                 LogbrokerTopicConfigResolver logbrokerTopicConfigResolver,
                                 SandboxResourcesResolver sandboxResolver, RelationController relationController,
                                 GlobalContext globalContext,
                                 SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher,
                                 SpecPatcher<DataModel.TEndpointSetSpec.Builder> endpointSetSpecPatcher,
                                 SpecPatcher<TDynamicResourceSpec.Builder> dynamicResourceSpecPatcher) {
        this.clock = clock;
        this.dockerImagesResolver = dockerImagesResolver;
        this.converter = converter;
        this.validator = validator;
        this.stageStatusSender = stageStatusSender;
        this.commonAclUpdater = commonAclUpdater;
        this.replicaSetAclUpdater = replicaSetAclUpdater;
        this.horizontalPodAutoscalerAclUpdater = horizontalPodAutoscalerAclUpdater;
        this.logbrokerTopicConfigResolver = logbrokerTopicConfigResolver;
        this.sandboxResolver = sandboxResolver;
        this.relationController = relationController;
        this.globalContext = globalContext;

        endpointSetManagers = getObjectManagers(ypClients, facadeFactory, YpObjectType.ENDPOINT_SET);
        autoscalerManagers = getObjectManagers(ypClients, facadeFactory, YpObjectType.HORIZONTAL_POD_AUTOSCALER);
        replicaSetManagers = getObjectManagers(ypClients, facadeFactory, YpObjectType.REPLICA_SET);
        dynamicResourceManagers = getObjectManagers(ypClients, facadeFactory, YpObjectType.DYNAMIC_RESOURCE);

        mcrsManager = facadeFactory.createManager(ClusterAndType.mcrs());

        this.aclFilter = aclFilter;
        this.podSpecPatcher = podSpecPatcher;
        this.endpointSetSpecPatcher = endpointSetSpecPatcher;
        this.dynamicResourceSpecPatcher = dynamicResourceSpecPatcher;
    }

    private static <Meta extends SchemaMeta, Spec extends Message, Status extends Message>
    Map<String, ObjectLifeCycleManager<Meta, Spec, Status>> getObjectManagers(
            AsyncYpClientsMap ypClients, ObjectLifeCycleManagerFactory facadeFactory, YpObjectType objectType) {
        return StreamEx.of(ypClients.getClusters())
                .toMap(cluster -> facadeFactory.createManager(ClusterAndType.perClusterInstance(cluster, objectType)));
    }

    public Set<ObjectLifeCycleManager> listObjectRepositories() {
        Set<ObjectLifeCycleManager> result = extractLifeCycleManagers(endpointSetManagers);
        result.add(mcrsManager);
        result.addAll(extractLifeCycleManagers(replicaSetManagers));
        result.addAll(extractLifeCycleManagers(autoscalerManagers));
        result.addAll(extractLifeCycleManagers(dynamicResourceManagers));
        return result;
    }

    @Override
    public DeployUnitController createUnitController(DeployUnitSpec spec,
                                                     String unitId,
                                                     String fullUnitId,
                                                     StageContext context,
                                                     Consumer<DeployUnitStatus> statusUpdateHandler,
                                                     BiConsumer<String, DynamicResourceStatus> dynamicResourceStatusHandler,
                                                     BiConsumer<String, DeployUnitStatus> handleUpdateEndpointSetStatus) {
        return new DeployUnitControllerImpl(spec, unitId, fullUnitId, context, this,
                this::createEndpointSetController,
                (String id, String cluster, Consumer<DynamicResourceRevisionStatus> updateNotifier) -> new DynamicResourceController(id,
                        dynamicResourceManagers.get(cluster), updateNotifier, fullUnitId,
                        converter),
                this::createHorizontalPodAutoscalerController, dockerImagesResolver,
                statusUpdateHandler, dynamicResourceStatusHandler,
                handleUpdateEndpointSetStatus, clock, logbrokerTopicConfigResolver,
                sandboxResolver, endpointSetSpecPatcher, dynamicResourceSpecPatcher);
    }

    @Override
    public StageControllerImpl createStageController(String id, Runnable updateHandler) {
        return new StageControllerImpl(id, this, commonAclUpdater, updateHandler, converter, globalContext);
    }

    @Override
    public ValidationController createValidationController(String stageId) {
        return new ValidationControllerImpl(this, stageId, converter, validator, clock, stageStatusSender, aclFilter, globalContext);
    }

    @Override
    public DeployPrimitiveController createDeployPrimitiveController(DeployUnitSpecDetails spec,
                                                                     Runnable statusUpdateHandler,
                                                                     String fullUnitId,
                                                                     String stageFqid) {
        if (spec instanceof ReplicaSetUnitSpec) {
            return new ReplicaSetDeployPrimitiveController(fullUnitId,
                    (id, cluster, updateNotifier) -> createSingleReplicaSetController(id, cluster, updateNotifier, stageFqid),
                    podSpecPatcher,
                    (v) -> statusUpdateHandler.run(),
                    converter);
        } else if (spec instanceof McrsUnitSpec) {
            MultiClusterReplicaSetController innerController = new MultiClusterReplicaSetController(fullUnitId,
                    mcrsManager,
                    (v) -> statusUpdateHandler.run(),
                    converter,
                    replicaSetAclUpdater,
                    stageFqid,
                    relationController);
            return new McrsDeployPrimitiveController(innerController, podSpecPatcher, converter);
        } else {
            throw new IllegalArgumentException(String.format("Cannot create controller for deploy unit class %s",
                    spec.getClass()));
        }
    }

    private ObjectController<TReplicaSetSpec, DeployPrimitiveStatus<TReplicaSetStatus>> createSingleReplicaSetController(String id,
                                                                                                                         String cluster,
                                                                                                                         Consumer<DeployPrimitiveStatus<TReplicaSetStatus>> updateNotifier,
                                                                                                                         String stageFqid) {
        return new SingleReplicaSetController(id,
                replicaSetManagers.get(cluster),
                updateNotifier,
                converter,
                replicaSetAclUpdater,
                stageFqid,
                relationController);
    }

    private ObjectController<DataModel.TEndpointSetSpec, ReadinessStatus>
    createEndpointSetController(String id, String cluster, Consumer<ReadinessStatus> updateNotifier) {
        return new EndpointSetController(id, endpointSetManagers.get(cluster),
                updateNotifier);
    }

    private ObjectController<THorizontalPodAutoscalerSpec, ReadinessStatus>
    createHorizontalPodAutoscalerController(String id, String cluster, Consumer<ReadinessStatus> updateNotifier) {
        return new HorizontalPodAutoscalerController(id, autoscalerManagers.get(cluster),
                updateNotifier,
                horizontalPodAutoscalerAclUpdater);
    }

    private static <Meta extends SchemaMeta, Spec extends Message, Status extends Message>
    Set<ObjectLifeCycleManager> extractLifeCycleManagers(Map<String, ObjectLifeCycleManager<Meta, Spec, Status>> facadeManagers) {
        return facadeManagers.values().stream().collect(Collectors.toSet());
    }
}
