package ru.yandex.infra.stage.inject;

import java.time.Clock;
import java.util.Map;

import com.google.protobuf.Message;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.yp.YpObjectSettings;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.yp.GarbageApprover;
import ru.yandex.infra.stage.yp.ObjectLifeCycleManagerImpl;
import ru.yandex.infra.stage.yp.RepositoryFactory;
import ru.yandex.yp.model.YpObjectType;

public class ObjectLifeCycleManagerFactory {

    public static final String METRIC_GARBAGE_LIMIT = "garbage_limit";
    public static final String METRIC_INITIAL_GARBAGE_LIMIT = "initial_garbage_limit";

    private final static Map<YpObjectType, String> METRIC_PREFIXES_BY_OBJECT_TYPE = Map.of(
            YpObjectType.ENDPOINT_SET, "endpoint_sets",
            YpObjectType.HORIZONTAL_POD_AUTOSCALER, "horizontal_pod_autoscalers",
            YpObjectType.REPLICA_SET, "replica_sets",
            YpObjectType.DYNAMIC_RESOURCE, "dynamic_resources",
            YpObjectType.RESOURCE_CACHE, "resource_caches",
            YpObjectType.MULTI_CLUSTER_REPLICA_SET, "mcrs"
    );

    private final SerialExecutor serialExecutor;
    private final GaugeRegistry gaugeRegistry;
    private final Clock clock;
    private final GarbageApprover approver;
    private final RepositoryFactory repositoryFactory;
    private final GCSettings gcSettings;
    private final Map<YpObjectType, YpObjectSettings> ypObjectsCacheSettings;

    public ObjectLifeCycleManagerFactory(SerialExecutor serialExecutor,
                                         GaugeRegistry gaugeRegistry,
                                         Clock clock,
                                         GarbageApprover approver,
                                         RepositoryFactory repositoryFactory,
                                         GCSettings gcSettings,
                                         Map<YpObjectType, YpObjectSettings> ypObjectsCacheSettings) {
        this.serialExecutor = serialExecutor;
        this.gaugeRegistry = gaugeRegistry;
        this.clock = clock;
        this.approver = approver;
        this.repositoryFactory = repositoryFactory;
        this.gcSettings = gcSettings;
        this.ypObjectsCacheSettings = ypObjectsCacheSettings;

        METRIC_PREFIXES_BY_OBJECT_TYPE.forEach((type, prefix) -> {
            GaugeRegistry metrics = new NamespacedGaugeRegistry(gaugeRegistry, prefix);
            GCLimit gcLimit = gcSettings.getGCLimit(type);
            metrics.add(METRIC_GARBAGE_LIMIT, new GolovanableGauge<>(gcLimit::getRegular, "axxx"));
            metrics.add(METRIC_INITIAL_GARBAGE_LIMIT, new GolovanableGauge<>(gcLimit::getInitial, "axxx"));
        });
    }

    public <Meta extends SchemaMeta, Spec extends Message, Status extends Message>
    ObjectLifeCycleManagerImpl<Meta, Spec, Status> createManager(ClusterAndType clusterAndType) {

        String gaugePrefix = METRIC_PREFIXES_BY_OBJECT_TYPE.get(clusterAndType.getType());
        if (clusterAndType.getCluster().isPresent()) {
            gaugePrefix = String.format("%s.%s", gaugePrefix, clusterAndType.getCluster().get());
        }
        GaugeRegistry namespacedRegistry = new NamespacedGaugeRegistry(gaugeRegistry, gaugePrefix);
        GCLimit gcLimit = gcSettings.getGCLimit(clusterAndType.getType());

        return new ObjectLifeCycleManagerImpl<>(repositoryFactory.create(clusterAndType), serialExecutor, namespacedRegistry, clock,
                approver, clusterAndType, gcLimit.getRegular(), gcLimit.getInitial(), clusterAndType.toString(),
                YpObjectSettings.getSettingsForType(ypObjectsCacheSettings, clusterAndType.getType()));
    }

}
