package ru.yandex.infra.stage.podspecs;

import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;

import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.Checksum;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.dto.SecretSelector;
import ru.yandex.infra.stage.dto.SidecarVolumeSettings;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.infra.stage.yp.Attributes;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TMonitoringInfo;
import ru.yandex.yp.client.api.TMonitoringWorkloadEndpoint;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EWorkloadTargetState;
import ru.yandex.yp.client.pods.LiteralEnvSelector;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TComputeResources;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.TEnvVarValue;
import ru.yandex.yp.client.pods.TFile;
import ru.yandex.yp.client.pods.TFiles;
import ru.yandex.yp.client.pods.TGenericVolume;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMutableWorkload;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TResourceMeta;
import ru.yandex.yp.client.pods.TSandboxResource;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TVerification;
import ru.yandex.yp.client.pods.TVolume;
import ru.yandex.yt.ytree.TAttribute;
import ru.yandex.yt.ytree.TAttributeDictionaryOrBuilder;

public class PodSpecUtils {
    private PodSpecUtils() {
    }

    public static final TAttribute USED_BY_INFRA_LABEL = Attributes.booleanAttribute("used_by_infra", true);
    // Size suffixes must be long to avoid int overflows in computations.
    // Some sizes are in integer, an explicit cast is appropriate there.
    public static final long KILOBYTE = 1024;
    public static final long MEGABYTE = 1024 * KILOBYTE;
    public static final long GIGABYTE = 1024 * MEGABYTE;
    public static final String SYSTEM_BOX_SPECIFIC_TYPE = "system";
    public static final String DEFAULT_BOX_SPECIFIC_TYPE = "default";
    public static final TAttribute DISABLE_DISK_ISOLATION_LABEL_ATTRIBUTE = TAttribute.newBuilder()
            .setKey("disable-disk-isolation-spi-15289")
            .setValue(ByteString.EMPTY)
            .build();

    public static void addBoxIfAbsent(TPodAgentSpec.Builder agentSpec, TBox.Builder box, String boxId,
                                      Optional<String> allocationId) {
        TBox.Builder boxBuilder = agentSpec.getBoxesBuilderList().stream()
                .filter(b -> b.getId().equals(boxId))
                .findFirst()
                .orElse(box);

        setDiskVolumeAllocationRef(boxBuilder, allocationId);
        if (agentSpec.getBoxesList().stream().noneMatch(b -> b.getId().equals(boxId))) {
            agentSpec.addBoxes(boxBuilder);
        }
    }

    public static void addLayerIfAbsent(TResourceGang.Builder resourcesBuilder, String layerId, TLayer.Builder layer,
                                        Optional<String> allocationId) {
        TLayer.Builder layerBuilder = resourcesBuilder.getLayersBuilderList().stream()
                .filter(l -> l.getId().equals(layerId))
                .findFirst()
                .orElse(layer);

        setDiskVolumeAllocationRef(layerBuilder, allocationId);
        if (resourcesBuilder.getLayersList().stream().noneMatch(r -> r.getId().equals(layerId))) {
            resourcesBuilder.addLayers(layerBuilder);
        }
    }

    public static void addVolumeIfAbsent(TPodAgentSpec.Builder agentSpec, String volumeId,
                                         Optional<String> allocationId) {
        TVolume.Builder volume = volumeWithAllocationRef(agentSpec, volumeId, allocationId);
        if (agentSpec.getVolumesList().stream().noneMatch(v -> v.getId().equals(volumeId))) {
            agentSpec.addVolumes(volume);
        }
    }

    public static void addStaticResourceIfAbsent(TResourceGang.Builder resourcesBuilder, TResource.Builder resource,
                                                 String resourceId, Optional<String> allocationId) {
        TResource.Builder confResource = resourcesBuilder.getStaticResourcesBuilderList().stream()
                .filter(r -> r.getId().equals(resourceId))
                .findFirst()
                .orElse(resource);

        setDiskVolumeAllocationRef(confResource, allocationId);
        if (resourcesBuilder.getStaticResourcesList().stream().noneMatch(r -> r.getId().equals(resourceId))) {
            resourcesBuilder.addStaticResources(confResource);
        }
    }

    public static void patchRootDiskVolumeRequests(DataModel.TPodSpec.Builder podSpec, long additionalCapacity) {
        podSpec.getDiskVolumeRequestsBuilderList().stream()
                .filter(request -> hasLabel(request.getLabelsBuilder(), USED_BY_INFRA_LABEL))
                .findAny()
                .ifPresent(r -> patchDiskVolumeRequest(r, additionalCapacity));
    }

    public static void patchDiskVolumeRequest(DataModel.TPodSpec.TDiskVolumeRequest.Builder request,
                                              long additionalCapacity) {
        if (request.hasQuotaPolicy()) {
            request.getQuotaPolicyBuilder().setCapacity(request.getQuotaPolicyBuilder().getCapacity() + additionalCapacity);
        } else if (request.hasExclusivePolicy()) {
            request.getExclusivePolicyBuilder().setMinCapacity(request.getExclusivePolicyBuilder().getMinCapacity() + additionalCapacity);
        } else {
            String message = String.format("Unknown capacity type for request %s", request.getId());
            throw new IllegalArgumentException(message);
        }
    }

    public static Optional<DataModel.TPodSpec.TDiskVolumeRequest.Builder> addSidecarDiskVolumeRequest(DataModel.TPodSpec.Builder podSpec, long capacity, SidecarDiskVolumeDescription diskVolumeDescription) {
        String diskVolumeId = diskVolumeDescription.getId();

        List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> requests = podSpec.getDiskVolumeRequestsBuilderList();

        Optional<DataModel.TPodSpec.TDiskVolumeRequest.Builder> request =
                requests.stream().filter(r -> r.getId().equals(diskVolumeId)).findFirst();

        if (request.isEmpty()) {
            DataModel.TPodSpec.TDiskVolumeRequest.Builder diskVolumeRequest = podSpec.addDiskVolumeRequestsBuilder();
            diskVolumeRequest.setId(diskVolumeId)
                    .setQuotaPolicy(
                            DataModel.TPodSpec.TDiskVolumeRequest.TQuotaPolicy.newBuilder()
                                    .setCapacity(capacity)
                                    .build())
                    .setStorageClass(diskVolumeDescription.getStorageClass());
            return Optional.of(diskVolumeRequest);
        }

        return Optional.empty();
    }

    public static TEnvVar literalEnvVar(String name, String value) {
        return TEnvVar.newBuilder()
                .setName(name)
                .setValue(TEnvVarValue.newBuilder()
                        .setLiteralEnv(LiteralEnvSelector.newBuilder()
                                .setValue(value)
                                .build())
                        .build())
                .build();
    }

    public static TEnvVar secretEnvVar(String name, SecretSelector secretSelector) {
        return TEnvVar.newBuilder()
                .setName(name)
                .setValue(TEnvVarValue.newBuilder()
                        .setSecretEnv(Converter.toProto(secretSelector))
                        .build())
                .build();
    }

    public static TUtilityContainer startContainer(String cmd) {
        return TUtilityContainer.newBuilder()
                .setCommandLine(cmd)
                .build();
    }

    public static TUtilityContainer startContainer(String startCmd, String coreCmd, int stdoutStderrLimitBytes) {
        return TUtilityContainer.newBuilder()
                .setCommandLine(startCmd)
                .setCoreCommand(coreCmd)
                .setStdoutAndStderrLimit(stdoutStderrLimitBytes)
                .build();
    }

    public static TUtilityContainer startContainer(String startCmd, String coreCmd, int stdoutStderrLimitBytes,
                                                   Optional<Path> stdoutFilePath, Optional<Path> stderrFilePath) {
        TUtilityContainer.Builder container = TUtilityContainer.newBuilder()
                .setCommandLine(startCmd)
                .setCoreCommand(coreCmd)
                .setStdoutAndStderrLimit(stdoutStderrLimitBytes);

        stdoutFilePath.ifPresent(p -> container.setStdoutFile(p.toString()));
        stderrFilePath.ifPresent(p -> container.setStderrFile(p.toString()));

        return container.build();
    }

    public static TUtilityContainer startContainer(String cmd, int stdoutStderrLimitBytes) {
        return TUtilityContainer.newBuilder()
                .setCommandLine(cmd)
                .setStdoutAndStderrLimit(stdoutStderrLimitBytes)
                .build();
    }

    public static TUtilityContainer startContainer(String cmd, AllComputeResources resources) {
        long cpu = resources.getVcpuLimit();
        long memory = resources.getMemoryLimit();
        return TUtilityContainer.newBuilder()
                .setComputeResources(TComputeResources.newBuilder()
                        .setVcpuGuarantee(cpu)
                        .setVcpuLimit(cpu)
                        .setMemoryGuarantee(memory)
                        .setMemoryLimit(memory)
                        .setAnonymousMemoryLimit(resources.getAnonymousMemoryLimit())
                        .build())
                .setCommandLine(cmd)
                .build();
    }

    public static DataModel.TPodSpec.TIP6AddressRequest.Builder ip6Request(String networkId, String vlanId) {
        return DataModel.TPodSpec.TIP6AddressRequest.newBuilder()
                .setNetworkId(networkId)
                .setVlanId(vlanId)
                .setEnableDns(true);
    }

    public static TResource.Builder staticResource(String resourceId, String url, String checksum) {
        return TResource.newBuilder()
                .setId(resourceId)
                .setUrl(url)
                .setVerification(TVerification.newBuilder()
                        .setChecksum(checksum)
                        .build());
    }

    public static TResource.Builder rawStaticResource(String resourceId, String fileName, String data) {
        return TResource.newBuilder()
                .setId(resourceId)
                .setFiles(TFiles.newBuilder()
                        .addFiles(TFile.newBuilder()

                                .setFileName(fileName)
                                .setRawData(data)
                                .build())
                        .build())
                .setVerification(TVerification.newBuilder()
                        .setChecksum(Checksum.EMPTY.toAgentFormat())
                        .build());
    }

    public static TResource.Builder staticResource(String id, Optional<DownloadableResource> optional,
                                                   ResourceWithMeta defaultValue) {
        DownloadableResource resource = optional.orElse(defaultValue.getResource());
        TResource.Builder builder = TResource.newBuilder()
                .setId(id)
                .setUrl(resource.getUrl())
                .setVerification(TVerification.newBuilder()
                        .setChecksum(resource.getChecksum().toAgentFormat()));
        defaultValue.getMeta().ifPresent(meta -> builder.setMeta(resourceMeta(meta)));
        return builder;
    }

    public static TLayer.Builder layer(String id, Optional<DownloadableResource> optional,
                                       ResourceWithMeta defaultValue) {
        DownloadableResource resource = optional.orElse(defaultValue.getResource());
        TLayer.Builder builder = TLayer.newBuilder()
                .setId(id)
                .setUrl(resource.getUrl())
                .setChecksum(resource.getChecksum().toAgentFormat());
        defaultValue.getMeta().ifPresent(meta -> builder.setMeta(resourceMeta(meta)));
        return builder;
    }

    public static TResourceMeta resourceMeta(SandboxResourceMeta meta) {
        return TResourceMeta.newBuilder()
                .setSandboxResource(TSandboxResource.newBuilder()
                        .setResourceId(String.valueOf(meta.getResourceId()))
                        .setTaskId(String.valueOf(meta.getTaskId())))
                .build();
    }

    // This method is public because it is used in EnvironmentValidator too.
    public static boolean hasLabel(TAttributeDictionaryOrBuilder labels, TAttribute label) {
        return labels.getAttributesList().contains(label);
    }

    public static Optional<String> allocationAccordingToDiskIsolationLabel(TPodTemplateSpec.Builder spec,
                                                                           Optional<String> allocation) {

        if (spec.getLabels().getAttributesList().stream().anyMatch(a -> a.getKey().equals(DISABLE_DISK_ISOLATION_LABEL_ATTRIBUTE.getKey()))) {
            return Optional.empty();
        }

        return allocation;
    }

    public static TMutableWorkload buildMutableWorkload(String workloadId) {
        return TMutableWorkload.newBuilder().setTargetState(EWorkloadTargetState.EWorkloadTarget_ACTIVE)
                .setWorkloadRef(workloadId).build();
    }

    public static TBox.Builder addResources(TBox.Builder box, AllComputeResources added,
                                            boolean patchBoxWithoutConstraints) {
        TComputeResources.Builder resource = box.getComputeResourcesBuilder();

        if (resource.getVcpuGuarantee() != 0 || patchBoxWithoutConstraints) {
            resource.setVcpuGuarantee(resource.getVcpuGuarantee() + added.getVcpuGuarantee());
        }
        if (resource.getMemoryGuarantee() != 0 || patchBoxWithoutConstraints) {
            resource.setMemoryGuarantee(resource.getMemoryGuarantee() + added.getMemoryGuarantee());
        }
        if (resource.getAnonymousMemoryLimit() != 0) {
            resource.setAnonymousMemoryLimit(resource.getAnonymousMemoryLimit() + added.getAnonymousMemoryLimit());
        }
        if (resource.getMemoryLimit() != 0) {
            resource.setMemoryLimit(resource.getMemoryLimit() + added.getMemoryLimit());
        }
        if (resource.getVcpuLimit() != 0) {
            resource.setVcpuLimit(resource.getVcpuLimit() + added.getVcpuLimit());
        }
        if (resource.getThreadLimit() != 0) {
            resource.setThreadLimit(resource.getThreadLimit() + added.getThreadLimit());
        }
        /* from pod_agent.proto
         * ------------------------------------------------------------------------
         *  // Квота на размер
         *  // NOTE: Не обрабатывается - porto не умеет во вложенные квоты по диску
         *  uint64 quota_bytes = 1;
         */
        return box;
    }

    public static void addResources(DataModel.TPodSpec.Builder podSpec,
                                    AllComputeResources added,
                                    Optional<SidecarDiskVolumeDescription> sidecarDiskVolumeDescription) {
        addCpuMemResources(podSpec, added);

        long diskCapacity = added.getDiskCapacity();
        sidecarDiskVolumeDescription.ifPresentOrElse((diskVolumeDescription) -> addSidecarDiskVolumeRequest(podSpec,
                        diskCapacity, diskVolumeDescription),
                () -> patchRootDiskVolumeRequests(podSpec, diskCapacity));
    }


    public static SidecarDiskVolumeDescription calculateSidecarDiskVolumeDescription(String stageId,
                                                                                     DataModel.TPodSpec.Builder podSpec, String sidecarDiskAllocationId, Optional<SidecarVolumeSettings> sidecarVolumeSettings, List<String> sidecarDiskAllocationIds) {

        //StageValidator guarantees that:
        //1. num of user disk requests >= 1
        //2. when num of user disks requests == 1 we can have empty sidecarVolumeSettings or if not empty -> storage
        // class != unrecognized
        //3. when num of user disks requests > 1 sidecarVolumeSettings are not empty and have hdd or ssd storage class
        //there is double check, same as in StageValidatorImpl
        List<DataModel.TPodSpec.TDiskVolumeRequest> userDiskRequests =
                podSpec.getDiskVolumeRequestsList().stream().filter(r -> !sidecarDiskAllocationIds.contains(r.getId())).collect(Collectors.toList());

        if (userDiskRequests.size() == 1) {
            String userDiskStorageClass = userDiskRequests.get(0).getStorageClass();
            String finalStorageClass = sidecarVolumeSettings.map(s -> {
                SidecarVolumeSettings.StorageClass storageClass = s.getStorageClass();
                if (storageClass == SidecarVolumeSettings.StorageClass.UNRECOGNIZED) {
                    throw new IllegalArgumentException(String.format("Unrecognized storage class in disk volume %s is" +
                            " not permitted, stage id: %s", sidecarDiskAllocationId, stageId));
                }

                if (storageClass == SidecarVolumeSettings.StorageClass.AUTO) {
                    return userDiskStorageClass;
                }
                //hdd or ssd
                return storageClass.name().toLowerCase();

            }).orElse(userDiskStorageClass);

            return new SidecarDiskVolumeDescription(sidecarDiskAllocationId, finalStorageClass);
        }

        if (userDiskRequests.size() > 1) {
            SidecarVolumeSettings.StorageClass storageClass = sidecarVolumeSettings.orElseThrow(() ->
                    new IllegalArgumentException(String.format("Empty storage class in disk volume %s is not " +
                            "permitted, stage id: %s", sidecarDiskAllocationId, stageId))
            ).getStorageClass();

            if (storageClass == SidecarVolumeSettings.StorageClass.UNRECOGNIZED) {
                throw new IllegalArgumentException(String.format("Unrecognized storage class in disk volume %s is not" +
                        " permitted, stage id: %s", sidecarDiskAllocationId, stageId));
            }

            if (storageClass == SidecarVolumeSettings.StorageClass.AUTO) {
                throw new IllegalArgumentException(String.format("Auto storage class in disk volume %s is not " +
                        "permitted with multiple user disks, stage id: %s", sidecarDiskAllocationId, stageId));
            }

            String finalStorageClass = sidecarVolumeSettings.get().getStorageClass().name().toLowerCase();
            return new SidecarDiskVolumeDescription(sidecarDiskAllocationId, finalStorageClass);
        }

        throw new IllegalArgumentException(String.format("0 number of user disks is not permittted, stage id: %s",
                stageId));
    }

    public static void addCpuMemResources(DataModel.TPodSpec.Builder podSpec,
                                          AllComputeResources added) {
        DataModel.TPodSpec.TResourceRequests.Builder resources = podSpec.getResourceRequestsBuilder();
        resources.setVcpuGuarantee(resources.getVcpuGuarantee() + added.getVcpuGuarantee());
        resources.setMemoryGuarantee(resources.getMemoryGuarantee() + added.getMemoryGuarantee());
        // limit == 0 -> unlimited
        if (resources.getAnonymousMemoryLimit() != 0) {
            resources.setAnonymousMemoryLimit(resources.getAnonymousMemoryLimit() + added.getAnonymousMemoryLimit());
        }
        if (resources.getMemoryLimit() != 0) {
            resources.setMemoryLimit(resources.getMemoryLimit() + added.getMemoryLimit());
        }
        if (resources.getVcpuLimit() != 0) {
            resources.setVcpuLimit(resources.getVcpuLimit() + added.getVcpuLimit());
        }
    }

    public static TVolume.Builder volumeWithAllocationRef(TPodAgentSpec.Builder agentSpec, String volumeId,
                                                          Optional<String> diskVolumeAllocationId) {
        TVolume.Builder volume = agentSpec.getVolumesBuilderList().stream()
                .filter(v -> v.getId().equals(volumeId)).findFirst()
                .orElse(TVolume.newBuilder()
                        .setId(volumeId)
                        .setGeneric(TGenericVolume.newBuilder().build()));

        diskVolumeAllocationId.ifPresent(volume::setVirtualDiskIdRef);
        return volume;
    }

    public static void setDiskVolumeAllocationRef(TBox.Builder box, Optional<String> diskVolumeAllocationId) {
        diskVolumeAllocationId.ifPresent(box::setVirtualDiskIdRef);
    }

    public static void setDiskVolumeAllocationRef(TLayer.Builder layer, Optional<String> diskVolumeAllocationId) {
        diskVolumeAllocationId.ifPresent(layer::setVirtualDiskIdRef);
    }

    public static void setDiskVolumeAllocationRef(TResource.Builder resource, Optional<String> diskVolumeAllocationId) {
        diskVolumeAllocationId.ifPresent(resource::setVirtualDiskIdRef);
    }

    public static void addSidecarMonitoring(TMonitoringInfo.Builder podMonitoringInfo, String workloadId,
                                            String sidecarItype) {
        podMonitoringInfo.addWorkloads(TMonitoringWorkloadEndpoint.newBuilder()
                .setWorkloadId(workloadId)
                .setInheritMissedLabels(true)
                .putAllLabels(ImmutableMap.of(MonitoringPatcherUtils.ITYPE_KEY, sidecarItype))
        );
    }

    @FunctionalInterface
    public interface SandboxResourceIdCalculator {
        Optional<Long> calculate(Optional<SandboxResourceInfo> resourceInfo,
                                 ResourceSupplier defaultResourceSupplier);
    }

    public static final SandboxResourceIdCalculator SANDBOX_RESOURCE_ID_CALCULATOR = (resourceInfo,
                                                                                      defaultResourceSupplier) ->
            resourceInfo.map(SandboxResourceInfo::getRevision)
                    .or(() -> defaultResourceSupplier.get().getMeta().map(SandboxResourceMeta::getResourceId));
}

