package ru.yandex.infra.stage.podspecs;

import java.time.Clock;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.typesafe.config.Config;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.testutil.FutureUtils;
import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.dto.Checksum;
import ru.yandex.infra.stage.dto.DownloadableResource;

// Suppliers configuration on program start
public class ResourceSupplierFactory {
    private static final String SANDBOX_INFO_KEY = "sandbox_info";

    private final SerialExecutor executor;
    private final SandboxReleaseGetter releaseGetter;
    private final Duration updateInterval;
    private final Duration retryOnStartInterval;
    private final Clock clock;
    private final GaugeRegistry registry;
    private final Map<String, ResourceSupplier> created = new HashMap<>();
    private final Map<Pair<String, Long>, ResourceSupplier> suppliers = new ConcurrentHashMap<>();
    private final Cache<ResourceWithMeta> persistence;

    public ResourceSupplierFactory(SerialExecutor executor, SandboxReleaseGetter releaseGetter, Duration updateInterval,
                                   Duration retryOnStartInterval, Clock clock, GaugeRegistry registry,
                                   Cache<ResourceWithMeta> persistance) {
        this.executor = executor;
        this.releaseGetter = releaseGetter;
        this.updateInterval = updateInterval;
        this.retryOnStartInterval = retryOnStartInterval;
        this.clock = clock;
        this.registry = registry;
        this.persistence = persistance;
    }

    public ResourceSupplier createIfAbsent(Config parentConfig, String resourceName, boolean useChecksum) {
        var supplier = created.get(resourceName);
        if (null == supplier) supplier = create(parentConfig, resourceName, useChecksum);
        return supplier;
    }

    // Accepts config with all resources and extract target one by resourceName
    // This way the client does not have to duplicate resourceName
    @VisibleForTesting
    public ResourceSupplier create(Config parentConfig, String resourceName, boolean useChecksum) {
        if (created.containsKey(resourceName)) {
            throw new IllegalArgumentException(String.format("Supplier for resource '%s' has already been created",
                    resourceName));
        }
        ResourceSupplier result = doCreate(parentConfig.getConfig(resourceName), resourceName, useChecksum);
        created.put(resourceName, result);
        return result;
    }

    public void startAll(Duration timeout) {
        List<CompletableFuture<?>> futures = created.values().stream()
                .map(ResourceSupplier::start)
                .collect(Collectors.toList());
        FutureUtils.getUnchecked(CompletableFuture.allOf(futures.toArray(new CompletableFuture[]{})), timeout);
        created.forEach((resourceName, supplier) -> registry.add(String.format("resources.%s.delay_seconds",
                resourceName),
                () -> supplier.timeSinceLastUpdate().toSeconds()));
    }

    public void validateAll(Duration timeout) {
        CompletableFuture<?>[] futures = created.values().stream()
                .map(ResourceSupplier::get)
                .map(r -> releaseGetter.validateSandboxResource(r.getMeta().get().getResourceId(),
                        r.getResource().getUrl()))
                .toArray(CompletableFuture[]::new);

        FutureUtils.getUnchecked(CompletableFuture.allOf(futures), timeout);
    }

    public Map<String, ResourceSupplier> getSuppliers() {
        return created;
    }

    private ResourceSupplier doCreate(Config config, String resourceName, boolean useChecksum) {
        String source = config.getString("source");
        switch (source) {
            case "sandbox":
                return new SandboxResourceSupplier(executor, releaseGetter, config.getString("resource_type"),
                        useChecksum, updateInterval, retryOnStartInterval, clock,
                        resourceName,
                        persistence, () -> { });
            case "config":
                String url = config.getString("url");
                String checksumKey = "checksum";
                Optional<SandboxResourceMeta> meta = config.hasPath(SANDBOX_INFO_KEY) ?
                        Optional.of(parseMeta(config.getConfig(SANDBOX_INFO_KEY))) : Optional.empty();
                if (useChecksum) {
                    return new FixedResourceSupplier(new DownloadableResource(url,
                            Checksum.fromString(config.getString(checksumKey))), meta, releaseGetter);
                } else {
                    if (config.hasPath(checksumKey)) {
                        throw new IllegalArgumentException(String.format("Checksum should not be set for resource '%s'", resourceName));
                    }
                    return new FixedResourceSupplier(new DownloadableResource(url, Checksum.EMPTY), meta, releaseGetter);
                }
            default:
                throw new IllegalArgumentException(String.format("Unknown source '%s' for resource '%s'", source, resourceName));
        }
    }

    private SandboxResourceMeta parseMeta(Config config) {
        Map<String, String> attributes =
                config.hasPath("attributes") ? config.getConfig("attributes").entrySet().stream()
                        .collect(Collectors.toMap(Map.Entry::getKey,
                                e -> (String) e.getValue().unwrapped())) : Collections.emptyMap();
        return new SandboxResourceMeta(config.getLong("task_id"), config.getLong("resource_id"), attributes);
    }
}
