package ru.yandex.infra.stage.podspecs;

import java.net.URLEncoder;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Charsets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.dto.Checksum;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.util.JsonHttpGetter;

public class SandboxReleaseGetterImpl implements SandboxReleaseGetter {
    private static final Logger LOG = LoggerFactory.getLogger(SandboxReleaseGetterImpl.class);
    private final String baseUrl;
    private final Map<String, String> attributes;
    private final String token;
    private final ObjectMapper mapper = new ObjectMapper();
    private final Map<Long, CompletableFuture<ResourceWithMeta>> resolvedReleases = new ConcurrentHashMap<>();
    private final Cache<ResourceWithMeta> cache;
    private final JsonHttpGetter jsonHttpGetter;

    public SandboxReleaseGetterImpl(String baseUrl, String token, Map<String, String> attributes,
                                    Cache<ResourceWithMeta> cache, JsonHttpGetter jsonHttpGetter) {
        this.baseUrl = baseUrl;
        this.attributes = attributes;
        this.token = token;
        this.cache = cache;
        this.jsonHttpGetter = jsonHttpGetter;

        cache.getAll().forEach((id, resource) -> {
            try {
                long resourceId = Long.parseLong(id);
                resolvedReleases.put(resourceId, CompletableFuture.completedFuture(resource));
            } catch (NumberFormatException ignore) { }
        });
    }

    @Override
    public CompletableFuture<ResourceWithMeta> getLatestRelease(String resourceType, boolean useChecksum) {
        try {
            ObjectNode attrsNode = mapper.createObjectNode();
            attributes.forEach(attrsNode::put);
            String attrsString = URLEncoder.encode(mapper.writeValueAsString(attrsNode), Charsets.UTF_8.name());
            String url = String.format("%s/api/v1.0/resource?limit=1&type=%s&attrs=%s&state=READY&order=-id",
                    baseUrl, resourceType, attrsString);
            return getResourceWithMeta(useChecksum, url, false);
        } catch (Exception e) {
            LOG.error("Sandbox request setup failed", e);
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<ResourceWithMeta> getReleaseByResourceId(long resourceId, boolean useChecksum) {

        CompletableFuture<ResourceWithMeta> futureFromCache = resolvedReleases.get(resourceId);
        if (futureFromCache != null) {
            return futureFromCache;
        }

        String url = String.format("%s/api/v1.0/resource/%d", baseUrl, resourceId);

        CompletableFuture<ResourceWithMeta> future = getResourceWithMeta(useChecksum, url, true);
        resolvedReleases.put(resourceId, future);

        future.whenComplete((resource, error) -> {
            if (error != null) {
                resolvedReleases.remove(resourceId);
            } else {
                cache.put(Long.toString(resourceId), resource);
            }
        });

        return future;
    }

    private CompletableFuture<ResourceWithMeta> getResourceWithMeta(boolean useChecksum, String url,
                                                                    boolean resourceById) {
        return jsonHttpGetter.get(url, token).thenApply(root -> {
            JsonNode releaseItem = resourceById ? root : root.get("items").get(0);
            Checksum checksum = useChecksum ? new Checksum(releaseItem.get("md5").textValue(), Checksum.Type.MD5)
                    : Checksum.EMPTY;
            DownloadableResource resource = new DownloadableResource(releaseItem.get("skynet_id").textValue(), checksum);
            Iterator<Map.Entry<String, JsonNode>> attributes = releaseItem.get("attributes").fields();
            Map<String, String> attributesMap = new TreeMap<>();
            while (attributes.hasNext()) {
                Map.Entry<String, JsonNode> next = attributes.next();
                attributesMap.put(next.getKey(), next.getValue().textValue() == null ?
                        next.getValue().toString() : next.getValue().textValue());
            }
            SandboxResourceMeta meta =
                    new SandboxResourceMeta(releaseItem.get("task").get("id").longValue(),
                            releaseItem.get("id").longValue(), attributesMap);
            return new ResourceWithMeta(resource, Optional.of(meta));
        });
    }

    @Override
    public CompletableFuture<Void> validateSandboxResource(long resourceId, String resourceUrl) {
        LOG.info("Validating sandbox resource {}", resourceId);

        String url = String.format("%s/api/v1.0/resource/%d", baseUrl, resourceId);

        return jsonHttpGetter.get(url, token).thenApply(root -> {
            if (!resourceUrl.equals(root.get("skynet_id").textValue())) {
                throw new RuntimeException(String.format("Different resource url in sandbox and config for resource %d", resourceId));
            }

            JsonNode attributes = root.get("attributes");
            if (!attributes.has("ttl")) {
                throw new RuntimeException(String.format("No ttl attribute for resource %d", resourceId));
            }

            if (!attributes.get("ttl").textValue().equals("inf")) {
                throw new RuntimeException(String.format("Incorrect ttl for resource %d. Accept only resource with inf ttl", resourceId));
            }

            return null;
        });
    }
}
