package ru.yandex.infra.stage.podspecs;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.cache.Cache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;

// Handles only periodic updates, no sandbox specifics (except resource type).
public class SandboxResourceSupplier implements ResourceSupplier {
    private static final Duration NOT_RETRIEVED_DELAY = Duration.ofHours(1);
    private static final Logger LOG = LoggerFactory.getLogger(SandboxResourceSupplier.class);
    private final SerialExecutor serialExecutor;
    private final Duration updateInterval;
    private final Duration retryOnStartInterval;
    private final SandboxReleaseGetter sandboxReleaseGetter;
    private final String resourceType;
    private final boolean useChecksum;
    private final Clock clock;
    private final Cache<ResourceWithMeta> persistence;
    private final Runnable updateAttemptCallback;      // for testing purposes
    private final String resourceName;

    private final CompletableFuture<?> started = new CompletableFuture<>();
    private volatile Optional<ResourceWithMeta> currentResource = Optional.empty();
    private volatile Optional<Instant> lastUpdateTime = Optional.empty();

    public SandboxResourceSupplier(SerialExecutor serialExecutor, SandboxReleaseGetter sandboxReleaseGetter, String resourceType,
                                   boolean useChecksum,
                                   Duration updateInterval, Duration retryOnStartInterval, Clock clock,
                                   String resourceName,
                                   Cache<ResourceWithMeta> persistence, Runnable updateAttemptCallback) {
        this.serialExecutor = serialExecutor;
        this.updateInterval = updateInterval;
        this.retryOnStartInterval = retryOnStartInterval;
        this.sandboxReleaseGetter = sandboxReleaseGetter;
        this.resourceType = resourceType;
        this.useChecksum = useChecksum;
        this.clock = clock;
        this.persistence = persistence;
        this.updateAttemptCallback = updateAttemptCallback;
        this.resourceName = resourceName;
    }

    @Override
    public ResourceWithMeta get() {
        String message = String.format("Sandbox resource %s has not been resolved yet", resourceType);
        return currentResource.orElseThrow(() -> new IllegalStateException(message));
    }

    // result completes after first successful resolve or if resource has been restored from persistent state
    @Override
    public CompletableFuture<?> start() {
        Optional<ResourceWithMeta> lastKnownResource = persistence.get(resourceName);
        if (lastKnownResource.isPresent()) {
            currentResource = lastKnownResource;
            lastUpdateTime = Optional.of(clock.instant());
            started.complete(null);
            serialExecutor.schedule(this::doGet, updateInterval);
        }
        else {
            serialExecutor.schedule(this::doGet, Duration.ZERO);
        }

        return started;
    }

    @Override
    public Duration timeSinceLastUpdate() {
        return lastUpdateTime.map(value -> Duration.between(value, clock.instant())).orElse(NOT_RETRIEVED_DELAY);
    }

    @Override
    public SandboxReleaseGetter getSandboxReleaseGetter() {
        return sandboxReleaseGetter;
    }

    private void doGet() {
        serialExecutor.submitFuture(sandboxReleaseGetter.getLatestRelease(resourceType, useChecksum),
                resource -> {
                    if (currentResource.map(value -> !value.equals(resource)).orElse(true)) {
                        try {
                            persistence.put(resourceName, resource)
                                    .get(updateInterval.dividedBy(2).toNanos(), TimeUnit.NANOSECONDS);
                        } catch (InterruptedException|ExecutionException|TimeoutException exception) {
                            LOG.error("Failed to put updated resource into cache: {}", resourceName, exception);
                            handleGetResult(null, exception);
                            return;
                        }
                    }
                    handleGetResult(resource, null);
                },
                error -> handleGetResult(null, error));
    }

    private void handleGetResult(ResourceWithMeta newResource, Throwable error) {
        try {
            if (error != null) {
                if (currentResource.isPresent()) {
                    LOG.warn("Failed to update {}: ", description(), error);
                    serialExecutor.schedule(this::doGet, updateInterval);
                } else {
                    LOG.warn("Failed to get initial version of {}: ", description(), error);
                    serialExecutor.schedule(this::doGet, retryOnStartInterval);
                }
                updateAttemptCallback.run();
            } else {
                lastUpdateTime = Optional.of(clock.instant());
                currentResource.ifPresentOrElse(
                        currentValue -> {
                            if (!currentValue.equals(newResource)) {
                                LOG.info("Got new version of {}: {}", description(), newResource);
                            } else {
                                LOG.debug("{} unchanged", description());
                            }
                        },
                        () -> LOG.info("Got initial version of {}: {}", description(), newResource)
                );
                currentResource = Optional.of(newResource);
                serialExecutor.schedule(this::doGet, updateInterval);
                // order is important, tests rely on attempt future completion before start() completion
                updateAttemptCallback.run();
                started.complete(null);
            }
        } catch (Exception e) {
            LOG.warn("Error handling {}: ", description(), e);
            updateAttemptCallback.run();
            serialExecutor.schedule(this::doGet, updateInterval);
        }
    }

    private String description() {
        return "Sandbox resource " + resourceType;
    }
}
