package ru.yandex.infra.stage.podspecs.patcher;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.typesafe.config.Config;

import ru.yandex.infra.stage.ConfigUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceSupplierFactory;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.coredump.CoredumpPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.coredump.CoredumpPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.default_anon_limit.AnonLimitPatcherContext;
import ru.yandex.infra.stage.podspecs.patcher.default_anon_limit.AnonLimitPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.docker.DockerPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.docker.DockerPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherConfig;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherConfigParser;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.portoworkload.PortoWorkloadPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.portoworkload.PortoWorkloadPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.sandbox.SandboxPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.sandbox.SandboxPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.security.FoldersLayerUrls;
import ru.yandex.infra.stage.podspecs.patcher.security.SecurityPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.security.SecurityPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.ThreadLimitsPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.ThreadLimitsPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.box.BoxThreadLimitsDistributorImpl;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.pod_agent.PodAgentThreadLimitPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.pod_agent.PodAgentThreadLimitPatcherV1Context;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherContexts;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV1Context;

public final class PatcherContextsFactory {

    public static PatcherContexts parseConfig(
            Config config,
            ResourceSupplierFactory factory,
            PatcherParameters patcherParameters
    ) {
        return new PatcherContextsFactory(
                config, ResourceSupplierFactoryHolder.with(factory), patcherParameters
        ).create();
    }

    // TODO add tests for createCommonEnvPatcherContexts
    @VisibleForTesting
    static final String ENV_VARS_OVERLOADED_BY_PORTO_CONFIG_PATH = "env_vars_overloaded_by_porto";

    @VisibleForTesting
    static final String STAGE_URL_LABEL_FORMAT_CONFIG_PATH = "stage_url_label_format";

    @VisibleForTesting
    static final String PATCH_BOX_SPECIFIC_TYPE_CONFIG_PATH = "patch_box_specific_type";

    @VisibleForTesting
    static final String COREDUMP_INSTANCECTL_BINARY_RESOURCE_NAME = "instancectl_binary";

    @VisibleForTesting
    static final String COREDUMP_GDB_LAYER_RESOURCE_NAME = "gdb_layer";

    @VisibleForTesting
    static final String PLACE_BINARY_REVISION_TO_POD_AGENT_META_CONFIG_PATH = "place_binary_revision_to_pod_agent_meta";

    @VisibleForTesting
    static final String DEFAULTS_POD_AGENT_BINARY_RESOURCE_NAME = "pod_agent_binary";

    @VisibleForTesting
    static final String DEFAULTS_POD_AGENT_LAYER_RESOURCE_NAME = "pod_agent_layer";

    @VisibleForTesting
    static final String DYNAMIC_RESOURCE_DRU_LAYER_RESOURCE_NAME = "dru_layer";

    @VisibleForTesting
    static final String JUGGLER_BINARY_RESOURCE_NAME = "juggler_binary";

    @VisibleForTesting
    static final String LOGBROKER_AGENT_LAYER_RESOURCE_NAME = "logbroker_agent_layer";

    @VisibleForTesting
    static final String LOGROTATE_BINARY_RESOURCE_NAME = "logrotate_binary";

    @VisibleForTesting
    static final String TVM_BASE_LAYER_RESOURCE_NAME = "tvm_base_layer";

    @VisibleForTesting
    static final String TVM_LAYER_RESOURCE_NAME = "tvm_layer";

    @VisibleForTesting
    static final String TVM_CONFIG_CONFIG_PATH = "tvm";

    @VisibleForTesting
    static final String TVM_DISK_SIZE_MB_CONFIG_PATH = "disk_size_mb";

    @VisibleForTesting
    static final String TVM_INSTALLATION_TAG_CONFIG_PATH = "installation_tag";

    @VisibleForTesting
    static final String SECURITY_CONFIG_PATH = "security";

    @VisibleForTesting
    static final String SECURITY_FIRST_AFFECTED_POD_AGENT_RESOURCE_ID_CONFIG_PATH = "first_affected_pod_agent_resource_id";

    @VisibleForTesting
    static final String SECURITY_POD_AGENT_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.pod_agent_folders_layer";
    @VisibleForTesting
    static final String SECURITY_PORTOSHELL_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.portoshell_folders_layer";
    @VisibleForTesting
    static final String SECURITY_COREDUMP_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.coredump_folders_layer";
    @VisibleForTesting
    static final String SECURITY_LOGBROKER_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.logbroker_folders_layer";
    @VisibleForTesting
    static final String SECURITY_LOGROTATE_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.logrotate_folders_layer";
    @VisibleForTesting
    static final String SECURITY_JUGGLER_FOLDERS_LAYER_URL_CONFIG_PATH = "layer_urls.juggler_folders_layer";

    private final Config config;
    private final ResourceSupplierFactoryHolder resourceSupplierFactoryHolder;
    private final PatcherParameters parameters;

    @VisibleForTesting
    PatcherContextsFactory(Config config,
                                  ResourceSupplierFactoryHolder resourceSupplierFactoryHolder,
                                  PatcherParameters parameters) {
        this.config = config;
        this.resourceSupplierFactoryHolder = resourceSupplierFactoryHolder;
        this.parameters = parameters;
    }

    @VisibleForTesting
    PatcherContexts create() {
        return PatcherContexts.with(
                createCommonEnvPatcherContexts(),
                createCoredumpPatcherContexts(),
                createDefaultsPatcherContexts(),
                createDockerPatcherContexts(),
                createDynamicResourcePatcherContexts(),
                createJugglerPatcherContexts(),
                createLogbrokerPatcherContexts(),
                createLogrotatePatcherContexts(),
                createMonitoringPatcherContexts(),
                createSandboxPatcherContexts(),
                createThreadLimitsPatcherContexts(),
                createTvmPatcherContexts(),
                createPortoWorkloadPatcherContexts(),
                createSecurityPatcherContexts(),
                createAnonLimitPatcherContexts(),
                createPodAgentThreadLimitPatcherContexts()
        );
    }

    private ResourceSupplier getResourceSupplier(String resourceName, boolean useChecksum) {
        return resourceSupplierFactoryHolder.createIfAbsent(config, resourceName, useChecksum);
    }

    private ResourceSupplier getResourceSupplierWithoutChecksum(String resourceName) {
        return getResourceSupplier(resourceName, false);
    }

    private ResourceSupplier getResourceSupplierWithChecksum(String resourceName) {
        return getResourceSupplier(resourceName, true);
    }

    private CommonEnvPatcherContexts createCommonEnvPatcherContexts() {
        return CommonEnvPatcherContexts.with(
                new CommonEnvPatcherV1Context(config.getStringList(ENV_VARS_OVERLOADED_BY_PORTO_CONFIG_PATH),
                        config.getString(STAGE_URL_LABEL_FORMAT_CONFIG_PATH))
        );
    }

    private CoredumpPatcherContexts createCoredumpPatcherContexts() {
        var instancectlSupplier = getResourceSupplierWithoutChecksum(COREDUMP_INSTANCECTL_BINARY_RESOURCE_NAME);
        var gdbSupplier = getResourceSupplierWithoutChecksum(COREDUMP_GDB_LAYER_RESOURCE_NAME);
        long releaseGetterTimeout = parameters.getReleaseGetterTimeoutSeconds();

        return CoredumpPatcherContexts.with(
                new CoredumpPatcherV1Context(
                        instancectlSupplier,
                        gdbSupplier,
                        releaseGetterTimeout
                )
        );
    }

    private DefaultsPatcherContexts createDefaultsPatcherContexts() {
        /*
        DefaultsPatcher.java (old)
        this.defaultPodAgentBinarySupplier = factory.create(config, "pod_agent_binary", true);
        this.defaultPodAgentLayerSupplier = factory.create(config, "pod_agent_layer", true);
         */
        var defaultPodAgentBinarySupplier = getResourceSupplierWithChecksum(DEFAULTS_POD_AGENT_BINARY_RESOURCE_NAME);
        var defaultPodAgentLayerSupplier = getResourceSupplierWithChecksum(DEFAULTS_POD_AGENT_LAYER_RESOURCE_NAME);

        Optional<String> podAgentAllocationId = parameters.getPodAgentAllocationId();

        boolean patchBoxSpecificType = config.getBoolean(PATCH_BOX_SPECIFIC_TYPE_CONFIG_PATH);

        boolean placeBinaryRevisionToPodAgentMeta = config.getBoolean(PLACE_BINARY_REVISION_TO_POD_AGENT_META_CONFIG_PATH);

        long releaseGetterTimeout = parameters.getReleaseGetterTimeoutSeconds();

        Optional<List<String>> allSidecarDiskAllocationIds = parameters.getAllSidecarDiskAllocationIds();

        return DefaultsPatcherContexts.with(
                new DefaultsPatcherV1Context(defaultPodAgentBinarySupplier, defaultPodAgentLayerSupplier,
                        podAgentAllocationId, patchBoxSpecificType,
                        releaseGetterTimeout, placeBinaryRevisionToPodAgentMeta, allSidecarDiskAllocationIds
                )
        );
    }

    private DockerPatcherContexts createDockerPatcherContexts() {
        return DockerPatcherContexts.with(
                new DockerPatcherV1Context()
        );
    }

    private DynamicResourcePatcherContexts createDynamicResourcePatcherContexts() {
        var druDefaultLayerSupplier = getResourceSupplierWithoutChecksum(DYNAMIC_RESOURCE_DRU_LAYER_RESOURCE_NAME);
        var druLayerSupplier = getResourceSupplierWithoutChecksum(DYNAMIC_RESOURCE_DRU_LAYER_RESOURCE_NAME);
        long releaseGetterTimeout = parameters.getReleaseGetterTimeoutSeconds();

        return DynamicResourcePatcherContexts.with(
                new DynamicResourcePatcherV1Context(
                        druDefaultLayerSupplier,
                        druLayerSupplier,
                        releaseGetterTimeout
                )
        );
    }

    private JugglerPatcherContexts createJugglerPatcherContexts() {
        var jugglerBinaryDefaultSupplier = getResourceSupplierWithoutChecksum(JUGGLER_BINARY_RESOURCE_NAME);

        return JugglerPatcherContexts.with(
                new JugglerPatcherV1Context(jugglerBinaryDefaultSupplier, parameters.getAllSidecarDiskAllocationIds())
        );
    }

    private LogbrokerPatcherContexts createLogbrokerPatcherContexts() {
        var logbrokerAgentLayerSupplier = getResourceSupplierWithoutChecksum(LOGBROKER_AGENT_LAYER_RESOURCE_NAME);

        Optional<String> logbrokerAllocationId = parameters.getLogbrokerAllocationId();

        Optional<List<String>> allSidecarDiskAllocationIds = parameters.getAllSidecarDiskAllocationIds();

        boolean patchBoxSpecificType = config.getBoolean(PATCH_BOX_SPECIFIC_TYPE_CONFIG_PATH);

        LogbrokerPatcherConfig logbrokerPatcherConfig = LogbrokerPatcherConfigParser.parseConfig(
                ConfigUtils.logbrokerPatcherConfig(config)
        );

        var unifiedAgentConfigFactory = logbrokerPatcherConfig.getUnifiedAgentConfigFactory();
        var boxResourcesConfig = logbrokerPatcherConfig.getBoxResourcesConfig();

        long releaseGetterTimeout = parameters.getReleaseGetterTimeoutSeconds();

        return LogbrokerPatcherContexts.with(
                new LogbrokerPatcherV1Context(
                        logbrokerAgentLayerSupplier,
                        logbrokerAllocationId, allSidecarDiskAllocationIds,
                        patchBoxSpecificType,
                        unifiedAgentConfigFactory,
                        boxResourcesConfig,
                        releaseGetterTimeout
                )
        );
    }

    private LogrotatePatcherContexts createLogrotatePatcherContexts() {
        var logrotateResourceSupplier = getResourceSupplierWithoutChecksum(LOGROTATE_BINARY_RESOURCE_NAME);

        return LogrotatePatcherContexts.with(
                new LogrotatePatcherV1Context(logrotateResourceSupplier)
        );
    }

    private MonitoringPatcherContexts createMonitoringPatcherContexts() {
        return MonitoringPatcherContexts.with(
                new MonitoringPatcherV1Context()
        );
    }

    private SandboxPatcherContexts createSandboxPatcherContexts() {
        return SandboxPatcherContexts.with(
                new SandboxPatcherV1Context()
        );
    }

    private ThreadLimitsPatcherContexts createThreadLimitsPatcherContexts() {
        return ThreadLimitsPatcherContexts.with(
                new ThreadLimitsPatcherV1Context(BoxThreadLimitsDistributorImpl.INSTANCE)
        );
    }

    private TvmPatcherContexts createTvmPatcherContexts() {
        Map<String, Integer> blackboxEnvironments = parameters.getBlackboxEnvironments();

        var baseLayerSupplier = getResourceSupplierWithoutChecksum(TVM_BASE_LAYER_RESOURCE_NAME);

        var tvmLayerSupplier = getResourceSupplierWithoutChecksum(TVM_LAYER_RESOURCE_NAME);

        var tvmConfig = config.getConfig(TVM_CONFIG_CONFIG_PATH);

        long diskSizeMb = tvmConfig.getLong(TVM_DISK_SIZE_MB_CONFIG_PATH);

        Optional<String> diskVolumeAllocationId = parameters.getTvmAllocationId();

        Optional<List<String>> allSidecarDiskAllocationIds = parameters.getAllSidecarDiskAllocationIds();

        String installationTag = tvmConfig.getString(TVM_INSTALLATION_TAG_CONFIG_PATH);

        boolean patchBoxSpecificType = config.getBoolean(PATCH_BOX_SPECIFIC_TYPE_CONFIG_PATH);

        long releaseGetterTimeout = parameters.getReleaseGetterTimeoutSeconds();

        return TvmPatcherContexts.with(
                new TvmPatcherV1Context(
                        blackboxEnvironments,
                        baseLayerSupplier, tvmLayerSupplier,
                        diskSizeMb,
                        diskVolumeAllocationId,
                        allSidecarDiskAllocationIds,
                        installationTag,
                        patchBoxSpecificType, releaseGetterTimeout
                )
        );
    }

    private PortoWorkloadPatcherContexts createPortoWorkloadPatcherContexts() {
        return PortoWorkloadPatcherContexts.with(new PortoWorkloadPatcherV1Context());
    }

    private SecurityPatcherContexts createSecurityPatcherContexts() {
        var defaultPodAgentBinarySupplier = getResourceSupplierWithChecksum(DEFAULTS_POD_AGENT_BINARY_RESOURCE_NAME);

        var securityConfig = config.getConfig(SECURITY_CONFIG_PATH);

        long firstAffectedPodAgentVersion = securityConfig.getLong(SECURITY_FIRST_AFFECTED_POD_AGENT_RESOURCE_ID_CONFIG_PATH);

        var foldersLayerUrls = new FoldersLayerUrls(
                securityConfig.getString(SECURITY_POD_AGENT_FOLDERS_LAYER_URL_CONFIG_PATH),
                securityConfig.getString(SECURITY_PORTOSHELL_FOLDERS_LAYER_URL_CONFIG_PATH),
                securityConfig.getString(SECURITY_COREDUMP_FOLDERS_LAYER_URL_CONFIG_PATH),
                securityConfig.getString(SECURITY_LOGBROKER_FOLDERS_LAYER_URL_CONFIG_PATH),
                securityConfig.getString(SECURITY_JUGGLER_FOLDERS_LAYER_URL_CONFIG_PATH),
                securityConfig.getString(SECURITY_LOGROTATE_FOLDERS_LAYER_URL_CONFIG_PATH)
        );

        return SecurityPatcherContexts.with(new SecurityPatcherV1Context(defaultPodAgentBinarySupplier, firstAffectedPodAgentVersion, foldersLayerUrls));
    }

    private AnonLimitPatcherContexts createAnonLimitPatcherContexts() {
        return AnonLimitPatcherContexts.with(new AnonLimitPatcherContext());
    }

    private PodAgentThreadLimitPatcherContexts createPodAgentThreadLimitPatcherContexts() {
        return PodAgentThreadLimitPatcherContexts.with(new PodAgentThreadLimitPatcherV1Context());
    }
}
