package ru.yandex.infra.stage.podspecs.patcher;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;
import ru.yandex.yp.client.pods.TBoxOrBuilder;

public class PatcherUtils {
    public static boolean isSystem(String boxId) {
        return Arrays.stream(SystemBoxId.values()).anyMatch(systemBoxId -> systemBoxId.getValue().equals(boxId));
    }

    public static boolean isSystem(TBoxOrBuilder box) {
        return isSystem(box.getId());
    }

    public static boolean notSystem(String boxId) {
        return !isSystem(boxId);
    }

    public static boolean notSystem(TBoxOrBuilder box) {
        return notSystem(box.getId());
    }

    public static boolean notLogbroker(String boxId) {
        switch (boxId) {
            case LogbrokerPatcherUtils.LOGBROKER_BOX_ID:
            case LogbrokerPatcherUtils.LOGBROKER_BOX_ID_UNDERSCORED:
                return false;
            default:
                return true;
        }
    }

    public static <T extends TBoxOrBuilder> List<T> notSystemBoxes(List<T> boxes) {
        return boxes.stream().filter(PatcherUtils::notSystem).collect(Collectors.toList());
    }

    public static ResourceWithMeta getResource(ResourceSupplier supplier,
                                               Optional<SandboxResourceInfo> sidecarInfo,
                                               long releaseGetterTimeoutSeconds) {
        AtomicReference<ResourceWithMeta> resourceWithMeta = new AtomicReference<>();
        resourceWithMeta.set(supplier.get());
        sidecarInfo.map(SandboxResourceInfo::getRevision)
                .filter(revision -> revision > 0)
                .ifPresent(revision -> {
                    CompletableFuture<ResourceWithMeta> releaseByResourceId =
                            supplier.getSandboxReleaseGetter().getReleaseByResourceId(revision, true);
                    try {
                        resourceWithMeta.set(releaseByResourceId.get(releaseGetterTimeoutSeconds, TimeUnit.SECONDS));
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    } catch (ExecutionException | TimeoutException e) {
                        throw new IllegalArgumentException("Sandbox resource not resolved: ", e);
                    }
                });

        return resourceWithMeta.get();
    }
}
