package ru.yandex.infra.stage.podspecs.patcher;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.coredump.CoredumpPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.coredump.CoredumpPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.coredump.CoredumpPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.default_anon_limit.AnonLimitPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.default_anon_limit.AnonLimitPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV4;
import ru.yandex.infra.stage.podspecs.patcher.defaults.DefaultsPatcherV5;
import ru.yandex.infra.stage.podspecs.patcher.docker.DockerPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.docker.DockerPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.docker.DockerPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.dynamic_resource.DynamicResourcePatcherV4;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV4;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV5;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV6;
import ru.yandex.infra.stage.podspecs.patcher.juggler.JugglerPatcherV7;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV4;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV5;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherV6;
import ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.network.NetworkPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.network.NetworkPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.network.NetworkPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.pod_agent.PodAgentPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.portoworkload.PortoWorkloadPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.sandbox.SandboxPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.security.SecurityPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.security.SecurityPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.ThreadLimitsPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.ThreadLimitsPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.thread_limits.pod_agent.PodAgentThreadLimitPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV1;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV2;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV3;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV4;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherV5;
import ru.yandex.yp.client.api.TPodTemplateSpec;

public final class PodSpecPatchersHolderFactory {

    public static PatchersHolder<TPodTemplateSpec.Builder> fromContexts(PatcherContexts contexts) {
        return new PodSpecPatchersHolderFactory(contexts).create();
    }

    private final PatcherContexts patcherContexts;
    private final Map<Class<? extends SpecPatcher<TPodTemplateSpec.Builder>>, SpecPatcher<TPodTemplateSpec.Builder>> classToPatcher;

    private PodSpecPatchersHolderFactory(PatcherContexts patcherContexts) {

        this.patcherContexts = patcherContexts;
        this.classToPatcher = new HashMap<>();
    }

    public PatchersHolder<TPodTemplateSpec.Builder> create() {
        initializePatchers();
        return new PatchersHolder<>(ImmutableMap.copyOf(classToPatcher));
    }

    private <PatcherContextType> void addPatchersWithContext(
            PatcherContextType patcherContext,
            Collection<Function<PatcherContextType, SpecPatcher<TPodTemplateSpec.Builder>>> patcherGenerators) {
        addPatchers(patcherGenerators.stream()
                .map(patcherGenerator -> patcherGenerator.apply(patcherContext))
        );
    }

    private void addPatchers(Stream<SpecPatcher<TPodTemplateSpec.Builder>> patchersStream) {
        patchersStream.forEach(this::addPatcher);
    }

    private void addPatcher(SpecPatcher<TPodTemplateSpec.Builder> patcher) {
        Class<? extends SpecPatcher<TPodTemplateSpec.Builder>> patcherClass =
                (Class<? extends SpecPatcher<TPodTemplateSpec.Builder>>) patcher.getClass();

        if (null != classToPatcher.get(patcherClass)) {
            throw new RuntimeException(
                    String.format(
                            "Patcher with class %s tried to instantiate twice",
                            patcherClass.getCanonicalName()
                    )
            );
        }

        classToPatcher.put(patcherClass, patcher);
    }

    private void initializePatchers() {
        initializeCommonEnvPatchers();
        initializeCoredumpPatchers();
        initializeDefaultsPatchers();
        initializeDockerPatchers();
        initializeDynamicResourcePatchers();
        initializeJugglerPatchers();
        initializeLogbrokerPatchers();
        initializeLogrotatePatchers();
        initializeMonitoringPatchers();
        initializeSandboxPatchers();
        initializeThreadLimitsPatchers();
        initializeTvmPatchers();
        initializeNetworkPatchers();
        initializePortoWorkloadPatchers();
        initializeSecurityPatchers();
        initializeAnonLimitPatchers();
        initializePodAgentPatchers();
        initializePodAgentThreadLimitPatchers();
    }

    private void initializeCommonEnvPatchers() {
        var commonEnvContexts = patcherContexts.getCommonEnvPatcherContexts();

        addPatchersWithContext(
                commonEnvContexts.getCommonEnvPatcherV1Context(),
                ImmutableList.of(CommonEnvPatcherV1::new, CommonEnvPatcherV2::new, CommonEnvPatcherV3::new));
    }

    private void initializeCoredumpPatchers() {
        var coredumpContexts = patcherContexts.getCoredumpPatcherContexts();

        addPatchersWithContext(
                coredumpContexts.getCoredumpPatcherV1Context(),
                ImmutableList.of(CoredumpPatcherV1::new, CoredumpPatcherV2::new, CoredumpPatcherV3::new)
        );
    }

    private void initializeDefaultsPatchers() {
        var defaultsContexts = patcherContexts.getDefaultsPatcherContexts();

        addPatchersWithContext(
                defaultsContexts.getDefaultsPatcherV1Context(),
                ImmutableList.of(DefaultsPatcherV1::new, DefaultsPatcherV2::new, DefaultsPatcherV3::new,
                        DefaultsPatcherV4::new, DefaultsPatcherV5::new)
        );
    }

    private void initializePodAgentPatchers() {
        addPatcher(new PodAgentPatcherV1());
    }

    private void initializeDockerPatchers() {
        var dockerContexts = patcherContexts.getDockerPatcherContexts();

        addPatchersWithContext(
                dockerContexts.getDockerPatcherV1Context(),
                ImmutableList.of(DockerPatcherV1::new, DockerPatcherV2::new, DockerPatcherV3::new)
        );
    }

    private void initializeDynamicResourcePatchers() {
        var dynamicResourcesContexts = patcherContexts.getDynamicResourcePatcherContexts();

        addPatchersWithContext(
                dynamicResourcesContexts.getDynamicResourcePatcherV1Context(),
                ImmutableList.of(DynamicResourcePatcherV1::new,
                        DynamicResourcePatcherV2::new,
                        DynamicResourcePatcherV3::new,
                        DynamicResourcePatcherV4::new)
        );
    }

    private void initializeJugglerPatchers() {
        var jugglerContexts = patcherContexts.getJugglerPatcherContexts();

        addPatchersWithContext(
                jugglerContexts.getJugglerPatcherV1Context(),
                ImmutableList.of(JugglerPatcherV1::new,
                        JugglerPatcherV2::new,
                        JugglerPatcherV3::new,
                        JugglerPatcherV4::new,
                        JugglerPatcherV5::new,
                        JugglerPatcherV6::new,
                        JugglerPatcherV7::new)
        );
    }

    private void initializeLogbrokerPatchers() {
        var logbrokerContexts = patcherContexts.getLogbrokerPatcherContexts();

        addPatchersWithContext(
                logbrokerContexts.getLogbrokerPatcherV1Context(),
                ImmutableList.of(
                        LogbrokerPatcherV1::new,
                        LogbrokerPatcherV2::new,
                        LogbrokerPatcherV3::new,
                        LogbrokerPatcherV4::new,
                        LogbrokerPatcherV5::new,
                        LogbrokerPatcherV6::new
                )
        );
    }

    private void initializeLogrotatePatchers() {
        var logrotateContexts = patcherContexts.getLogrotatePatcherContexts();

        addPatchersWithContext(
                logrotateContexts.getLogrotatePatcherV1Context(),
                ImmutableList.of(
                        LogrotatePatcherV1::new,
                        LogrotatePatcherV2::new
                )
        );
    }

    private void initializeMonitoringPatchers() {
        var monitoringContexts = patcherContexts.getMonitoringPatcherContexts();

        addPatchersWithContext(
                monitoringContexts.getMonitoringPatcherV1Context(),
                ImmutableList.of(MonitoringPatcherV1::new, MonitoringPatcherV2::new, MonitoringPatcherV3::new)
        );
    }

    private void initializeSandboxPatchers() {
        var sandboxContexts = patcherContexts.getSandboxPatcherContexts();

        addPatchersWithContext(
                sandboxContexts.getSandboxPatcherV1Context(),
                ImmutableList.of(SandboxPatcherV1::new)
        );
    }

    private void initializeThreadLimitsPatchers() {
        var threadLimitsContexts = patcherContexts.getThreadLimitsPatcherContexts();

        addPatchersWithContext(
                threadLimitsContexts.getThreadLimitsPatcherV1Context(),
                ImmutableList.of(ThreadLimitsPatcherV1::new,
                        ThreadLimitsPatcherV2::new)
        );
    }

    private void initializeTvmPatchers() {
        var tvmContexts = patcherContexts.getTvmPatcherContexts();

        addPatchersWithContext(
                tvmContexts.getTvmPatcherV1Context(),
                ImmutableList.of(TvmPatcherV1::new,
                        TvmPatcherV2::new,
                        TvmPatcherV3::new,
                        TvmPatcherV4::new,
                        TvmPatcherV5::new)
        );
    }

    private void initializeNetworkPatchers() {
        addPatcher(new NetworkPatcherV1());
        addPatcher(new NetworkPatcherV2());
        addPatcher(new NetworkPatcherV3());
    }

    private void initializePortoWorkloadPatchers() {
        var portoWorkloadContexts = patcherContexts.getPortoWorkloadPatcherContexts();

        addPatchersWithContext(
                portoWorkloadContexts.getPortoWorkloadPatcherV1Context(),
                ImmutableList.of(PortoWorkloadPatcherV1::new)
        );
    }

    private void initializeSecurityPatchers() {
        var securityContexts = patcherContexts.getSecurityPatcherContexts();

        addPatchersWithContext(
                securityContexts.getSecurityPatcherV1Context(),
                ImmutableList.of(SecurityPatcherV1::new, SecurityPatcherV2::new)
        );
    }

    private void initializeAnonLimitPatchers() {
        var contexts = patcherContexts.getAnonLimitPatcherContexts();

        addPatchersWithContext(
                contexts.getAnonLimitPatcherContext(),
                ImmutableList.of(AnonLimitPatcherV1::new, AnonLimitPatcherV2::new)
        );
    }

    private void initializePodAgentThreadLimitPatchers() {
        var podAgentThreadLimitPatcherContexts = patcherContexts.getPodAgentThreadLimitPatcherContexts();

        addPatchersWithContext(
                podAgentThreadLimitPatcherContexts.getPodAgentThreadLimitPatcherV1Context(),
                ImmutableList.of(PodAgentThreadLimitPatcherV1::new)
        );
    }
}
