package ru.yandex.infra.stage.podspecs.patcher.common_env;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.TPodAgentSpec;

import static ru.yandex.infra.stage.podspecs.patcher.PatcherUtils.notLogbroker;
import static ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils.DEPLOY_BOX_ID_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils.DEPLOY_PROJECT_ID_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils.DEPLOY_STAGE_ID_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils.DEPLOY_UNIT_ID_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.common_env.CommonEnvPatcherUtils.DEPLOY_WORKLOAD_ID_ENV_NAME;

// Sets common env(env vars + labels)
abstract class CommonEnvPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {

    @VisibleForTesting
    static final String DEPLOY_UNIT_ID_LABEL_KEY = "deploy_unit_id";

    @VisibleForTesting
    static final String STAGE_ID_LABEL_KEY = "stage_id";

    @VisibleForTesting
    static final String STAGE_URL_LABEL_KEY = "stage_url";

    @VisibleForTesting
    static final String PROJECT_ID_LABEL_KEY = "project_id";

    final CommonEnvPatcherV1Context context;

    public CommonEnvPatcherV1Base(CommonEnvPatcherV1Context context) {
        this.context = context;
    }

    public void patchV1(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        String stageId = context.getStageContext().getStageId();
        String deployUnitId = context.getDeployUnitId();
        String projectId = context.getStageContext().getProjectId();

        patchCommonEnvVars(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder(),
                            stageId,
                            deployUnitId,
                            projectId,
                            context.getStageContext());

        commonLabels(stageId, deployUnitId, projectId, labelsBuilder);
    }

    private static void patchCommonEnvVars(TPodAgentSpec.Builder agentSpec,
                                           String stageId,
                                           String deployUnitId,
                                           String projectId,
                                           StageContext stageContext) {
        //for logbroker we have personal env vars
        agentSpec.getBoxesBuilderList().stream()
                .filter(boxBuilder -> notLogbroker(boxBuilder.getId()))
                .forEach(boxBuilder -> {
                    boxBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_BOX_ID_ENV_NAME, boxBuilder.getId()))
                              .addEnv(PodSpecUtils.literalEnvVar(DEPLOY_UNIT_ID_ENV_NAME, deployUnitId))
                              .addEnv(PodSpecUtils.literalEnvVar(DEPLOY_STAGE_ID_ENV_NAME, stageId))
                              .addEnv(PodSpecUtils.literalEnvVar(DEPLOY_PROJECT_ID_ENV_NAME, projectId));

                    Set<String> allVarsInBox = boxBuilder.getEnvList()
                                                         .stream()
                                                         .map(TEnvVar::getName)
                                                         .collect(Collectors.toSet());
                    stageContext.getEnvVars().forEach((key, value) -> {
                        if(!allVarsInBox.contains(key)) {
                            boxBuilder.addEnv(PodSpecUtils.literalEnvVar(key, value));
                        }
                    });
                });

        agentSpec.getWorkloadsBuilderList().stream()
                .filter(workload -> notLogbroker(workload.getBoxRef()))
                .forEach(workload -> workload.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_WORKLOAD_ID_ENV_NAME,
                        workload.getId())));
    }

    public void commonLabels(String stageId, String deployUnitId, String projectId, YTreeBuilder labelsBuilder) {
        labelsBuilder.key(DEPLOY_UNIT_ID_LABEL_KEY).value(deployUnitId)
                .key(STAGE_ID_LABEL_KEY).value(stageId)
                .key(PROJECT_ID_LABEL_KEY).value(projectId)
                .key(STAGE_URL_LABEL_KEY).value(String.format(context.stageUrlLabelFormat, stageId));
    }

    public void patchV2(TPodTemplateSpec.Builder builder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        patchV1(builder, context, labelsBuilder);
        patchResourceRequestEnvVars(builder);
    }

    public void patchV3(TPodTemplateSpec.Builder builder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        patchV2(builder, context, labelsBuilder);
        patchEnvVarsOverloadedByPorto(builder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder());
    }

    private static void patchResourceRequestEnvVars(TPodTemplateSpec.Builder builder) {
        final DataModel.TPodSpec.Builder specBuilder = builder.getSpecBuilder();
        TPodAgentSpec.Builder agentSpec = specBuilder.getPodAgentPayloadBuilder().getSpecBuilder();

        DataModel.TPodSpec.TResourceRequests.Builder podResources = specBuilder.getResourceRequestsBuilder();
        agentSpec.getBoxesBuilderList()
                .stream()
                .filter(PatcherUtils::notSystem)
                .forEach(boxBuilder -> {
                    var boxResources = boxBuilder.getComputeResourcesBuilder();
                    for (var resource : ResourceRequestParameter.values()) {
                        long boxValue = resource.getComputeResourceValue.apply(boxResources);
                        long value = boxValue != 0 ? boxValue : resource.getResourceRequestValue.apply(podResources);
                        boxBuilder.addEnv(PodSpecUtils.literalEnvVar(resource.getEnvVarName(), Long.toString(value)));
                    }
                });

        agentSpec.getWorkloadsBuilderList()
                .stream()
                .filter(workload -> PatcherUtils.notSystem(workload.getBoxRef()))
                .forEach(workload -> {
                    var workloadResources = workload.getStartBuilder().getComputeResourcesBuilder();
                    for (var resource : ResourceRequestParameter.values()) {
                        long value = resource.getComputeResourceValue.apply(workloadResources);
                        if (value != 0) {
                            workload.addEnv(PodSpecUtils.literalEnvVar(resource.getEnvVarName(), Long.toString(value)));
                        }
                    }
                });
    }

    /*
        Fix inheritance of env variables from the box to the workflow overwritten by Porto
        https://bb.yandex-team.ru/projects/PORTO/repos/porto/browse/src/container.cpp#2436
     */
    private void patchEnvVarsOverloadedByPorto(TPodAgentSpec.Builder agentSpec) {
        List<String> envVarsOverloadedByPortoNames = context.envVarsOverloadedByPortoNames;

        Map<String, Map<String, String>> boxIdToVars = new HashMap<>();

        agentSpec.getBoxesList().forEach(box -> {
            Map<String, String> varsInBox = box.getEnvList().stream()
                    .collect(Collectors.toMap(
                            TEnvVar::getName,
                            envVar -> envVar.getValue().getLiteralEnv().getValue())
                    );
            boxIdToVars.put(box.getId(), varsInBox);
        });

        agentSpec.getWorkloadsBuilderList()
                .forEach(workloadBuilder -> {
                    Set<String> varsInWorkload = workloadBuilder
                            .getEnvList().stream()
                            .map(TEnvVar::getName)
                            .collect(Collectors.toSet());

                    Map<String, String> varsInBox = boxIdToVars.get(workloadBuilder.getBoxRef());

                    for (String varName : envVarsOverloadedByPortoNames) {
                        String valueInBox = varsInBox.getOrDefault(varName, null);
                        if (valueInBox != null && !varsInWorkload.contains(varName)) {
                            workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(varName, valueInBox));
                        }
                    }
                });
    }
}
