package ru.yandex.infra.stage.podspecs.patcher.coredump;

import java.nio.file.Path;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.CoredumpConfig;
import ru.yandex.infra.stage.dto.CoredumpOutputPolicy;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMountedStaticResource;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TUlimitSoft;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.MEGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addLayerIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addStaticResourceIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addVolumeIfAbsent;
import static ru.yandex.yp.client.pods.EContainerULimitType.EContainerULimit_CORE;

abstract class CoredumpPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    private static final String DEFAULT_AGGR_URL = "https://coredumps.n.yandex-team.ru/submit_core";
    private static final String COREDUMPS_UTIL_PATH = "/coredumps_bin";

    @VisibleForTesting
    static final String COREDUMPS_PATH = "/coredumps_box";

    private static final String GDB_PATH = "/gdb/bin/gdb";
    private static final String INSTANCECTL_BINARY_DEFAULT_ID = "instancectl-binary";
    private static final String GDB_LAYER = "gdb-layer";

    protected final ResourceSupplier instancectlSupplier;
    protected final ResourceSupplier gdbSupplier;
    protected final long releaseGetterTimeoutSeconds;

    public CoredumpPatcherV1Base(CoredumpPatcherV1Context context) {
        this(context.getInstancectlSupplier(), context.getGdbSupplier(), context.getReleaseGetterTimeoutSeconds());
    }

    public CoredumpPatcherV1Base(ResourceSupplier instancectlSupplier, ResourceSupplier gdbSupplier,
                                 long releaseGetterTimeoutSeconds) {
        this.instancectlSupplier = instancectlSupplier;
        this.gdbSupplier = gdbSupplier;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context,
                      YTreeBuilder labelsBuilder) {

        Map<String, CoredumpConfig> coredumpConfigs = context.getSpec().getCoredumpConfig();
        TPodAgentSpec.Builder agentSpec =
                podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder();
        String stageId = context.getStageContext().getStageId();
        HashSet<String> boxToPatch = new HashSet<>();
        HashSet<String> boxToPatchingWithGDB = new HashSet<>();

        agentSpec.getWorkloadsBuilderList().forEach(workload -> {
            String workloadId = workload.getId();
            CoredumpConfig config = coredumpConfigs.get(workload.getId());
            if (config == null) {
                return;
            }

            getUlimitSoftCoredumpIndex(workload.getUlimitSoftList()).ifPresent(workload::removeUlimitSoft);

            workload.addUlimitSoft(TUlimitSoft.newBuilder()
                    .setName(EContainerULimit_CORE)
                    .setValue(config.getTotalSizeLimit() * MEGABYTE)
                    .build());

            TUtilityContainer.Builder start = workload.getStartBuilder();

            Optional<String> itype = getItype(podTemplateSpecBuilder, workloadId);
            String coredumpOutputPath = getCoredumpOutputPath(config.getOutputPolicy(), workload, stageId, agentSpec);
            start.setCoreCommand(buildCoreCommand(config, stageId, workloadId, coredumpOutputPath, itype));
            boxToPatch.add(workload.getBoxRef());
            if (config.isAggregatorEnabled()) {
                boxToPatchingWithGDB.add(workload.getBoxRef());
            }
        });

        if (boxToPatch.isEmpty()) {
            return;
        }

        patchBoxes(agentSpec, context, boxToPatch, boxToPatchingWithGDB);
    }

    protected ResourceWithMeta getInstancectl(DeployUnitContext context) {
        return PatcherUtils.getResource(instancectlSupplier, context.getSpec().getCoredumpToolResourceInfo(),
                releaseGetterTimeoutSeconds);
    }

    protected ResourceWithMeta getGdb(DeployUnitContext context) {
        return PatcherUtils.getResource(gdbSupplier, context.getSpec().getGdbLayerResourceInfo(),
                releaseGetterTimeoutSeconds);
    }

    private void patchBoxes(TPodAgentSpec.Builder agentSpec,
                            DeployUnitContext context,
                            HashSet<String> boxToPatch,
                            HashSet<String> boxToPatchingWithGDB) {
        TResourceGang.Builder resourceBuilder = agentSpec.getResourcesBuilder();

        agentSpec.getBoxesBuilderList().stream()
                .filter(box -> boxToPatch.contains(box.getId()))
                .forEach(box -> {
                    if (boxToPatchingWithGDB.contains(box.getId())) {
                        addGDBLayer(box, context, resourceBuilder);
                    }

                    String virtualDiskIdRef = box.getVirtualDiskIdRef();

                    String volumeRef = String.format("%s_%s", "COREDUMP_VOLUME", virtualDiskIdRef);

                    addVolumeIfAbsent(agentSpec, volumeRef, Optional.of(virtualDiskIdRef));
                    box.addVolumes(
                            TMountedVolume.newBuilder()
                                    .setVolumeRef(volumeRef)
                                    .setMountPoint(COREDUMPS_PATH)
                                    .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                    .build());

                    String instanceCtlResourceId = String.format("%s_%s", INSTANCECTL_BINARY_DEFAULT_ID,
                            virtualDiskIdRef);
                    addStaticResourceIfAbsent(resourceBuilder,
                            PodSpecUtils.staticResource(instanceCtlResourceId,
                                    Optional.empty(),
                                    getInstancectl(context)),
                            instanceCtlResourceId,
                            Optional.of(virtualDiskIdRef));

                    box.addStaticResources(TMountedStaticResource.newBuilder()
                            .setMountPoint(COREDUMPS_UTIL_PATH)
                            .setResourceRef(instanceCtlResourceId));
                });
    }

    public static Optional<Integer> getUlimitSoftCoredumpIndex(List<TUlimitSoft> ulimitSoftList) {
        Optional<Integer> ulimitSoftCoredumpIndex = Optional.empty();
        for (int index = 0; index < ulimitSoftList.size(); index++) {
            if (ulimitSoftList.get(index).getName().equals(EContainerULimit_CORE)) {
                ulimitSoftCoredumpIndex = Optional.of(index);
            }
        }
        return ulimitSoftCoredumpIndex;
    }

    private static String getCoredumpOutputPath(Optional<CoredumpOutputPolicy> outputPolicy,
                                                TWorkload.Builder workload,
                                                String stageId,
                                                TPodAgentSpec.Builder agentSpec) {
        return outputPolicy.map(outPolicy -> {
            if (outPolicy.isOutputPath()) {
                return outPolicy.getValue();
            }

            // out volume from config used
            if (outPolicy.isOutputVolumeId()) {
                String volumeMountPoint = getVolumeMountPoint(agentSpec, workload.getBoxRef(), outPolicy.getValue());

                return coredumpPathWhileVolumeUsed(volumeMountPoint, stageId, workload.getId());
            }
            return null;
            // standard out volume used
        }).orElse(coredumpPathWhileVolumeUsed(COREDUMPS_PATH, stageId, workload.getId()));
    }

    private static String getVolumeMountPoint(TPodAgentSpec.Builder agentSpec, String boxId, String outVolumeId) {
        return agentSpec.getBoxesList().stream()
                .filter(b -> b.getId().equals(boxId))
                .flatMap(b -> b.getVolumesList().stream())
                .filter(v -> v.getVolumeRef().equals(outVolumeId))
                .findFirst()
                .map(TMountedVolume::getMountPoint)
                .orElseThrow();
    }

    private static String coredumpPathWhileVolumeUsed(String volumeMountPoint, String stageId, String workloadId) {
        return Path.of(volumeMountPoint, String.format("%s_%s", stageId, workloadId)).toString();
    }

    private static String buildCoreCommand(CoredumpConfig config,
                                           String stageId,
                                           String workloadId,
                                           String coredumpOutputPath,
                                           Optional<String> itypeMonitoring) {
        if (config.getProbability() < 0 || config.getProbability() > 100) {
            throw new IllegalArgumentException(String.format("Invalid coredump probability %d",
                    config.getProbability()));
        }
        StringJoiner command = new StringJoiner(" ");
        // base instance ctl core process command
        command.add(COREDUMPS_UTIL_PATH + "/instancectl core_process");

        command.add(String.format("--output %s", coredumpOutputPath));
        command.add(String.format("--instance-name=%s", "${DEPLOY_POD_PERSISTENT_FQDN}"));
        command.add(String.format("--svc-name=%s", config.getServiceName()
                .orElse(itypeMonitoring.orElse(getDefaultItype(stageId, workloadId)))));
        command.add(String.format("--probability=%d", config.getProbability()));
        command.add(String.format("--count-limit=%d", config.getCountLimit()));
        command.add(String.format("--total-size-limit=%d", config.getTotalSizeLimit() * MEGABYTE));
        config.getCtype().ifPresent(ctype -> command.add(String.format("--ctype=%s", ctype)));

        // send coredump to aggregator
        if (config.isAggregatorEnabled()) {
            String url = config.getAggregatorUrl();
            command.add(String.format("--aggr-url=%s", StringUtils.isNoneBlank(url) ? url : DEFAULT_AGGR_URL));
            command.add(String.format("--gdb-path=%s", GDB_PATH));
        }

        // setup cleanup policy
        if (config.getTtlSeconds() != 0) {
            command.add(String.format("--ttl=%d", config.getTtlSeconds()));
        }

        return command.toString();
    }

    @NotNull
    public static String getDefaultItype(String stageId, String workloadId) {
        return "deploy." + stageId + "." + workloadId;
    }

    private void addGDBLayer(TBox.Builder box,
                             DeployUnitContext context,
                             TResourceGang.Builder resourceBuilder) {
        String layerId = String.format("%s_%s", GDB_LAYER, box.getVirtualDiskIdRef());
        TLayer.Builder gdbLayer = PodSpecUtils.layer(layerId, Optional.empty(), getGdb(context));
        addLayerIfAbsent(resourceBuilder, layerId, gdbLayer, Optional.of(box.getVirtualDiskIdRef()));
        TRootfsVolume.Builder layerRefs = box.getRootfsBuilder().addLayerRefs(layerId);
        box.setRootfs(layerRefs);
    }

    protected Optional<String> getItype(TPodTemplateSpec.Builder podTemplateSpecBuilder, String workloadId) {
        var monitoring = podTemplateSpecBuilder.getSpecBuilder().getHostInfra().getMonitoring();
        var workload = monitoring.getWorkloadsList().stream()
                .filter(it -> it.getWorkloadId().equals(workloadId))
                .findFirst();
        return Optional.ofNullable(workload.map(it -> it.getLabelsMap().get("itype"))
                .orElse(monitoring.getLabelsMap().get("itype")));
    }

}
